/**
 *  Copyright (c) 2012 BMW
 *
 *  \author Christian Mueller, christian.ei.mueller@bmw.de BMW 2011,2012
 *
 *  \copyright
 *  Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction,
 *  including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so,
 *  subject to the following conditions:
 *  The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 *  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR
 *  THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 *  For further information see http://www.genivi.org/.
 *
 *  Modified  on: Oct 30, 2013
 *  Author: Vyankatesh VD
 */

#include "control/IAmControlSend.h"

#include <cassert>
#include <algorithm>
#include "shared/CAmDltWrapper.h"
#include "control/IAmControlReceive.h"
#include "AudioStack/clStackRules.h"
#include "AudioStack/clGeniviAudioCtrlAdapter.h"
#include "common.h"

#include "amcontrol_clienthandler_fc_audioroutemanager.h"
#include "amcontrol_clienthandler_fc_audiomanager.h"
#include "CAmControlSenderBase.h"


#define CFC_FI_S_IMPORT_INTERFACE_FI_TYPES
#include "cfc_fi_if.h"


/* Include message framework (AMT, msgfw) */
#define FI_S_IMPORT_INTERFACE_FI_MESSAGE
/* Include AudioRoute Manager FI interface */
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_MASCFFI_FUNCTIONIDS
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_MASCFFI_TYPES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_MASCFFI_ERRORCODES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_MASCFFI_SERVICEINFO
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_AUDIO_ROUTE_MGRFI_FUNCTIONIDS
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_AUDIO_ROUTE_MGRFI_TYPES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_AUDIO_ROUTE_MGRFI_ERRORCODES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_AUDIO_ROUTE_MGRFI_SERVICEINFO
#include "midw_fi_if.h"

#include "inc_fi_handler.h"

#define G3G_NUMBER_SYSTEM_SND_PROPERTY (SYP_MAX-1)
#define G3G_DEFAULT_SND_PROPERTY_VALUE 5

using namespace am;
using namespace AudioStack;

DLT_DECLARE_CONTEXT(ControllerPlugin)
DLT_DECLARE_CONTEXT(amcpincccalib)

tCString trc_szGetFileName(tCString szFilePathName)
{
 //  assert(false);
   return szFilePathName;
}

/******************************GLOBAL DATA*********************************/
uint8_t g_u8SystemSoundPropertiesId[G3G_NUMBER_SYSTEM_SND_PROPERTY]={
                               SYP_BEEP,SYP_FADER_STATUS_ACTIVE,SYP_VOLMODE_STREAM_MAIN_1,SYP_VOLMODE_STREAM_MIX_1
                              };
/**************************************************************************/

/************************************************************************
* FUNCTION        : g3g_amcontrollerplugin_soFactory
* DESCRIPTION     : Called by Audio Daemon while loading  plugin
* PARAMETER       :
* RETURNVALUE     :
* HISTORY         : GENIVI
************************************************************************/
extern "C" IAmControlSend* g3g_amcontrollerplugin_soFactory()
{
    return (new CAmControlSenderBase());
}

/************************************************************************
* FUNCTION        : destroyControlPluginInterface
* DESCRIPTION     : Called by Audio Daemon while unloading  plugin
* PARAMETER       :
* RETURNVALUE     :
* HISTORY         : GENIVI
************************************************************************/
extern "C" void destroyControlPluginInterface(IAmControlSend* controlSendInterface)
{
    delete controlSendInterface;
}

/************************************************************************
* FUNCTION        : CAmControlSenderBase
* DESCRIPTION     : Contructor
* PARAMETER       :
* RETURNVALUE     :
* HISTORY         : GENIVI
************************************************************************/
CAmControlSenderBase::CAmControlSenderBase() :
        mControlReceiveInterface(NULL), // /*  mListOpenConnections(), */
        mListMainSystemSoundPropertyChanges(),
        mListMainSinkSoundPropertyChanges(),
        mListOpenVolumeChanges(), //
        mListOpenMuteDemuteChanges(),
      mConnectSf (SFC_RAMP_DOWN), //
      g_domainId(0), //
      m_aSourceMap(), //
      m_aSinkMap(), //
      mConnectData(), //
        mStateflow(SF_NONE), //
        mClientHandlerFCAudioMgr(NULL), //
        m_pSocketHandler(NULL), //
        mClientHandlerAudioRouteMgr(NULL), //
        mControlReceiveShadowInterface(NULL)
{

   CAmDltWrapper::instance()->registerContext(ControllerPlugin, "AMCP", "RoutingPlugin for vehicle Domain");
   CAmDltWrapper::instance()->registerContext(amcpincccalib, "CPIL", "Controller Plugin INC CCA Interface");

   mListMainSourceSoundPropertyChanges.clear();
   mListMainSinkSoundPropertyChanges.clear();
   mListMainSystemSoundPropertyChanges.clear();

   //Activate logging in audio util lib
   inc_fi_init_dlt(amcpincccalib);
   log(&amcpincccalib,DLT_LOG_INFO,"DLT Context initialized for INC CCA LIB");
}

/************************************************************************
* FUNCTION        : ~CAmControlSenderBase
* DESCRIPTION     : Desructor
* PARAMETER       :
* RETURNVALUE     :
* HISTORY         : GENIVI
************************************************************************/
CAmControlSenderBase::~CAmControlSenderBase()
{
   log(&ControllerPlugin,DLT_LOG_INFO,"Entered in CAmControlSenderBase::~CAmControlSenderBase");
   if(E_OK == DeRegistersAll()){
      log(&ControllerPlugin,DLT_LOG_INFO,"Sources, Sinks and Domain are Deregistered!");
   }
   else{
      log(&ControllerPlugin,DLT_LOG_INFO,"Failed to Deregistered Sources, Sinks and Domain !");
   }
   mControlReceiveInterface = NULL;
}

/************************************************************************
 *FUNCTION     : DeRegistersAll
 *DESCRIPTION  : Function to deregister all source,sinks and domain
 *PARAMETER    : NA
 *RETURN VALUE : am_Error_e
 *HISTORY      :
 *17.09.2013   Rev 1.0     RBEI/ECG4 - Vyankatesh VD  Initial Revision
************************************************************************/
am_Error_e CAmControlSenderBase::DeRegistersAll()
{
   log(&ControllerPlugin,DLT_LOG_INFO,"Entered in DeRegistersAll");

   std::map<uint16_t, am::am_Source_s>::iterator itSource;
   std::map<uint16_t, am::am_Sink_s>::iterator itSink;

   am_Source_s rSource;
   am_Sink_s rSink;
   am_Error_e error = E_UNKNOWN;

   for (itSource = m_aSourceMap.begin(); itSource !=m_aSourceMap.end(); ++itSource)
   {
      rSource = itSource->second;
      error = hookSystemDeregisterSource(rSource.sourceID);
   }

   for (itSink = m_aSinkMap.begin(); itSink !=m_aSinkMap.end(); ++itSink)
   {
      rSink = itSink->second;
      error = hookSystemDeregisterSink(rSink.sinkID);
   }

   error = hookSystemDeregisterDomain(g_domainId);

   return error;
}

/************************************************************************
* FUNCTION     : startupController
* DESCRIPTION  :
* PARAMETER    : IAmControlReceive*
* RETURNVALUE  : am_Error_e
* HISTORY      :
*     04.09.12 Vyankatesh VD  Initial Revision.
************************************************************************/
am_Error_e CAmControlSenderBase::startupController(IAmControlReceive *controlreceiveinterface)
{
   log(&ControllerPlugin,DLT_LOG_INFO,"Entered in CAmControlSenderBase::startupController");

   if(NULL == controlreceiveinterface)
      return E_UNKNOWN;

   mControlReceiveInterface = controlreceiveinterface;
   mControlReceiveInterface->getSocketHandler(m_pSocketHandler);

   log(&ControllerPlugin,DLT_LOG_INFO,"Entered in CAmControlSenderBase::startupController");
   mControlReceiveShadowInterface = new IAmControlReceiverShadow(mControlReceiveInterface,m_pSocketHandler);

   if(mControlReceiveShadowInterface)
   {
      mClientHandlerAudioRouteMgr = new amcontrol_clienthandler_fc_audioroutemanager(this,mControlReceiveShadowInterface);
      mClientHandlerFCAudioMgr = new amcontrol_clienthandler_fc_audiomanager(this,mControlReceiveShadowInterface);
   }

   log(&ControllerPlugin,DLT_LOG_INFO,"Entered in CAmControlSenderBase::startupController, returning");

   return E_OK;
}

/************************************************************************
* FUNCTION     : setControllerReady
* DESCRIPTION  : To set Controller plugin ready
* PARAMETER    :
* RETURNVALUE  : void
* HISTORY      :
*     04.09.12 Vyankatesh VD  Initial Revision.
************************************************************************/
void CAmControlSenderBase::setControllerReady()
{
   log(&ControllerPlugin,DLT_LOG_INFO,"Entered in CAmControlSenderBase::setControllerReady");

   if(mControlReceiveInterface)
   {
      log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::setControllerReady, calling setRoutingReady");
      //here is a good place to insert Source and SinkClasses into the database...
      mControlReceiveInterface->setRoutingReady();
      mControlReceiveInterface->setCommandReady();

      //Enter System properties in DB
      std::vector<am_SystemProperty_s> listSystemSoundProperties;
      am_SystemProperty_s systemSoundProperty;
      systemSoundProperty.value = G3G_DEFAULT_SND_PROPERTY_VALUE;

      for (uint16_t i16Index = 0; i16Index < G3G_NUMBER_SYSTEM_SND_PROPERTY ; i16Index++)
      {
         systemSoundProperty.type = (am_SystemPropertyType_e)g_u8SystemSoundPropertiesId[i16Index];
         listSystemSoundProperties.push_back(systemSoundProperty);
      }

      log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::setControllerReady, calling enterSystemPropertiesListDB");
      mControlReceiveInterface->enterSystemPropertiesListDB(listSystemSoundProperties);
   }
}

/************************************************************************
* FUNCTION     : InitStackManager
* DESCRIPTION  : To initialize StackManager
* PARAMETER    :
* RETURNVALUE  : void
* HISTORY      :
*     15.10.13 Vyankatesh VD  Initial Revision.
************************************************************************/
void CAmControlSenderBase::InitStackManager()
{
  clGeniviAudioCtrlAdapter::Init(mControlReceiveInterface, mClientHandlerAudioRouteMgr);
}

/************************************************************************
* FUNCTION     : hookUserConnectionRequest
* DESCRIPTION  : to establish new connection
* PARAMETER    : const am_sourceID_t,const am_sinkID_t, am_mainConnectionID_t &,
* RETURNVALUE  : am_Error_e
* HISTORY      :
*     04.09.12 Vyankatesh VD  Initial Revision.
************************************************************************/
am_Error_e CAmControlSenderBase::hookUserConnectionRequest(const am_sourceID_t sourceID, const am_sinkID_t sinkID, am_mainConnectionID_t & mainConnectionID)
{
   logInfo("CAmControlSenderBase::hookUserConnectionRequest entered");
   if(mControlReceiveShadowInterface)
   {
     mControlReceiveShadowInterface->m_bUserRequest = true;
     mControlReceiveShadowInterface->enRequestType = REQUEST_SOURCE_ROUTE_ACTIVITY;
   }

   return clGeniviAudioCtrlAdapter::hookUserConnectionRequest(sourceID,sinkID,mainConnectionID);
}

/************************************************************************
* FUNCTION     : hookUserDisconnectionRequest
* DESCRIPTION  : to disconnect the existing connection
* PARAMETER    : const am_mainConnectionID_t
* RETURNVALUE  : am_Error_e
* HISTORY      :
*     04.09.12 Vyankatesh VD  Initial Revision.
************************************************************************/
am_Error_e CAmControlSenderBase::hookUserDisconnectionRequest(const am_mainConnectionID_t connectionID)
{
   log(&ControllerPlugin,DLT_LOG_INFO,"CAmRoutingSenderAsync::asyncDisconnect,Calling clGeniviAudioCtrlAdapter::hookUserConnectionRequest:");
   return clGeniviAudioCtrlAdapter::hookUserDisconnectionRequest(connectionID);
}

/************************************************************************
* FUNCTION     : vChangeSinkSoundProperty_Bass
* DESCRIPTION  : to call fc_audioMgr function to set bass
* PARAMETER    : rMainSinkSoundPropertySet&,uint16_t
* RETURNVALUE  : void
* HISTORY      :
*     20.09.13 Vyankatesh VD  Initial Revision.
************************************************************************/
void CAmControlSenderBase::vChangeSinkSoundProperty_Bass(rMainSinkSoundPropertySet & set,uint16_t ui16ProType)
{
   if((NULL == mControlReceiveShadowInterface) || (NULL == mClientHandlerFCAudioMgr)){
      log(&ControllerPlugin,DLT_LOG_INFO,"vChangeSinkSoundProperty_Bass,NULL pointers");
      return;
   }

   if(MSP_SOUND_BASS == ui16ProType)
   {
      mControlReceiveShadowInterface->enRequestType = REQUEST_MSP_BASS;
      log(&ControllerPlugin,DLT_LOG_INFO,"vChangeSinkSoundProperty_Bass Called for MSP_SOUND_BASS");
      mClientHandlerFCAudioMgr->vSetSinkSoundProperty_Bass(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_ABS,midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS,int8_t(set.mainSoundProperty.value));
   }
   else
   {
      set.mainSoundProperty.type = MSP_SOUND_MIDDLE;
      mControlReceiveShadowInterface->enRequestType = REQUEST_MSP_BASS;
      if(set.mainSoundProperty.value > 0){
         log(&ControllerPlugin,DLT_LOG_INFO,"vChangeSinkSoundProperty_Bass Called for MSP_SOUND_BASS_RELATIVE with increament");
         mClientHandlerFCAudioMgr->vSetSinkSoundProperty_Bass(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_INC,midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS,int8_t(set.mainSoundProperty.value));
      }
      else{
         log(&ControllerPlugin,DLT_LOG_INFO,"vChangeSinkSoundProperty_Bass Called for MSP_SOUND_BASS_RELATIVE with decreament by:",abs(set.mainSoundProperty.value));
         mClientHandlerFCAudioMgr->vSetSinkSoundProperty_Bass(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_DEC,midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS,abs(set.mainSoundProperty.value));
      }
   }
}

/************************************************************************
* FUNCTION     : vChangeSinkSoundProperty_Treble
* DESCRIPTION  : to call fc_audioMgr function to set treble
* PARAMETER    : rMainSinkSoundPropertySet&,uint16_t
* RETURNVALUE  : void
* HISTORY      :
*     20.09.13 Vyankatesh VD  Initial Revision.
************************************************************************/
void CAmControlSenderBase::vChangeSinkSoundProperty_Treble(rMainSinkSoundPropertySet & set,uint16_t ui16ProType)
{
   if((NULL == mControlReceiveShadowInterface) || (NULL == mClientHandlerFCAudioMgr)){
      log(&ControllerPlugin,DLT_LOG_INFO,"vChangeSinkSoundProperty_Treble,NULL pointers");
      return;
   }

   if(MSP_SOUND_TREBLE == ui16ProType)
   {
      mControlReceiveShadowInterface->enRequestType = REQUEST_MSP_TREBLE;
      log(&ControllerPlugin,DLT_LOG_INFO,"vChangeSinkSoundProperty_Treble Called for MSP_SOUND_TREBLE");
      mClientHandlerFCAudioMgr->vSetSinkSoundProperty_Treble(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_ABS,midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS,set.mainSoundProperty.value);
   }
   else
   {
      set.mainSoundProperty.type = MSP_SOUND_TREBLE;
      mControlReceiveShadowInterface->enRequestType = REQUEST_MSP_TREBLE;
      if(set.mainSoundProperty.value > 0){
         log(&ControllerPlugin,DLT_LOG_INFO,"vChangeSinkSoundProperty_Treble Called for MSP_SOUND_TREBLE_RELATIVE with increament");
         mClientHandlerFCAudioMgr->vSetSinkSoundProperty_Treble(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_INC,midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS,set.mainSoundProperty.value);
      }
      else{
         log(&ControllerPlugin,DLT_LOG_INFO,"vChangeSinkSoundProperty_Treble Called for MSP_SOUND_TREBLE_RELATIVE with decreament by:",abs(set.mainSoundProperty.value));
         mClientHandlerFCAudioMgr->vSetSinkSoundProperty_Treble(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_DEC,midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS,abs(set.mainSoundProperty.value));
      }
   }
}

/************************************************************************
* FUNCTION     : vChangeSinkSoundProperty_MidRange
* DESCRIPTION  : to call fc_audioMgr function to set midrange
* PARAMETER    : rMainSinkSoundPropertySet&,uint16_t
* RETURNVALUE  : void
* HISTORY      :
*     20.09.13 Vyankatesh VD  Initial Revision.
************************************************************************/
void CAmControlSenderBase::vChangeSinkSoundProperty_MidRange(rMainSinkSoundPropertySet & set,uint16_t ui16ProType)
{
   if((NULL == mControlReceiveShadowInterface) || (NULL == mClientHandlerFCAudioMgr)){
      log(&ControllerPlugin,DLT_LOG_INFO,"vChangeSinkSoundProperty_MidRange,NULL pointers");
      return;
   }
   if(MSP_SOUND_MIDDLE == ui16ProType)
   {
      mControlReceiveShadowInterface->enRequestType = REQUEST_MSP_MIDRANGE;
      log(&ControllerPlugin,DLT_LOG_INFO,"vChangeSinkSoundProperty_MidRange Called for MSP_SOUND_MIDDLE");
      mClientHandlerFCAudioMgr->vSetSinkSoundProperty_MidRange(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_ABS,midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS,set.mainSoundProperty.value);
   }
   else
   {
      set.mainSoundProperty.type = MSP_SOUND_TREBLE;
      mControlReceiveShadowInterface->enRequestType = REQUEST_MSP_MIDRANGE;
      if(set.mainSoundProperty.value > 0){
         log(&ControllerPlugin,DLT_LOG_INFO,"vChangeSinkSoundProperty_MidRange Called for MSP_SOUND_MIDDLE_RELATIVE with increament");
         mClientHandlerFCAudioMgr->vSetSinkSoundProperty_MidRange(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_INC,midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS,set.mainSoundProperty.value);
      }
      else{
         log(&ControllerPlugin,DLT_LOG_INFO,"vChangeSinkSoundProperty_MidRange Called for MSP_SOUND_MIDDLE_RELATIVE with decreament by:",abs(set.mainSoundProperty.value));
         mClientHandlerFCAudioMgr->vSetSinkSoundProperty_MidRange(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_DEC,midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS,abs(set.mainSoundProperty.value));
      }
   }
}

/************************************************************************
* FUNCTION     : vChangeSinkSoundProperty_Loudness
* DESCRIPTION  : to call fc_audioMgr function to set Loudness
* PARAMETER    : rMainSinkSoundPropertySet&,uint16_t
* RETURNVALUE  : void
* HISTORY      :
*     20.09.13 Vyankatesh VD  Initial Revision.
************************************************************************/
void CAmControlSenderBase::vChangeSinkSoundProperty_Loudness(rMainSinkSoundPropertySet & set,uint16_t ui16ProType)
{
   if((NULL == mControlReceiveShadowInterface) || (NULL == mClientHandlerFCAudioMgr)){
      log(&ControllerPlugin,DLT_LOG_INFO,"vChangeSinkSoundProperty_Loudness,NULL pointers");
      return;
   }
   if(MSP_SOUND_LOUDNESS == ui16ProType)
   {
      mControlReceiveShadowInterface->enRequestType = REQUEST_MSP_LOUDNESS;
      log(&ControllerPlugin,DLT_LOG_INFO,"vChangeSinkSoundProperty_Loudness Called for MSP_SOUND_LOUDNESS");
      mClientHandlerFCAudioMgr->vSetSinkSoundProperty_Loudness(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_ABS,midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS,set.mainSoundProperty.value);
   }
   else
   {
      set.mainSoundProperty.type = MSP_SOUND_LOUDNESS;
      mControlReceiveShadowInterface->enRequestType = REQUEST_MSP_LOUDNESS;
      if(set.mainSoundProperty.value > 0){
         log(&ControllerPlugin,DLT_LOG_INFO,"vChangeSinkSoundProperty_Loudness Called for MSP_SOUND_LOUDNESS_RELATIVE with increament");
         mClientHandlerFCAudioMgr->vSetSinkSoundProperty_Loudness(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_INC,midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS,set.mainSoundProperty.value);
      }
      else{
         log(&ControllerPlugin,DLT_LOG_INFO,"vChangeSinkSoundProperty_Loudness Called for MSP_SOUND_LOUDNESS_RELATIVE with decreament by:",abs(set.mainSoundProperty.value));
         mClientHandlerFCAudioMgr->vSetSinkSoundProperty_Loudness(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_DEC,midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS,abs(set.mainSoundProperty.value));
      }
   }
}

/************************************************************************
* FUNCTION     : vChangeSinkSoundProperty_Balance
* DESCRIPTION  : to call fc_audioMgr function to set Balance
* PARAMETER    : rMainSinkSoundPropertySet&,uint16_t
* RETURNVALUE  : void
* HISTORY      :
*     20.09.13 Vyankatesh VD  Initial Revision.
************************************************************************/
void CAmControlSenderBase::vChangeSinkSoundProperty_Balance(rMainSinkSoundPropertySet & set,uint16_t ui16ProType)
{
   if((NULL == mControlReceiveShadowInterface) || (NULL == mClientHandlerFCAudioMgr)){
      log(&ControllerPlugin,DLT_LOG_INFO,"vChangeSinkSoundProperty_Balance,NULL pointers");
      return;
   }
   if(MSP_SOUND_BALANCE == ui16ProType)
   {
      mControlReceiveShadowInterface->enRequestType = REQUEST_MSP_BALANCE;
      log(&ControllerPlugin,DLT_LOG_INFO,"vChangeSinkSoundProperty_Balance Called for MSP_SOUND_BALANCE");
      mClientHandlerFCAudioMgr->vSetSinkSoundProperty_Balance(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_ABS,midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS,set.mainSoundProperty.value);
   }
   else
   {
      set.mainSoundProperty.type = MSP_SOUND_BALANCE;
      mControlReceiveShadowInterface->enRequestType = REQUEST_MSP_BALANCE;
      if(set.mainSoundProperty.value > 0){
         log(&ControllerPlugin,DLT_LOG_INFO,"vChangeSinkSoundProperty_Balance Called for MSP_SOUND_LOUDNESS_RELATIVE with increament");
         mClientHandlerFCAudioMgr->vSetSinkSoundProperty_Balance(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_INC,midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS,set.mainSoundProperty.value);
      }
      else{
         log(&ControllerPlugin,DLT_LOG_INFO,"vChangeSinkSoundProperty_Balance Called for MSP_SOUND_LOUDNESS_RELATIVE with decreament by:",abs(set.mainSoundProperty.value));
         mClientHandlerFCAudioMgr->vSetSinkSoundProperty_Balance(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_DEC,midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS,abs(set.mainSoundProperty.value));
      }
   }
}
/************************************************************************
* FUNCTION     : vChangeSinkSoundProperty_Fader
* DESCRIPTION  : to call fc_audioMgr function to set Fader
* PARAMETER    : rMainSinkSoundPropertySet&,uint16_t
* RETURNVALUE  : void
* HISTORY      :
*     20.09.13 Vyankatesh VD  Initial Revision.
************************************************************************/
void CAmControlSenderBase::vChangeSinkSoundProperty_Fader(rMainSinkSoundPropertySet & set,uint16_t ui16ProType)
{
   if((NULL == mControlReceiveShadowInterface) || (NULL == mClientHandlerFCAudioMgr)){
      log(&ControllerPlugin,DLT_LOG_INFO,"vChangeSinkSoundProperty_Fader,NULL pointers");
      return;
   }
   if(MSP_SOUND_FADER == ui16ProType)
   {
      mControlReceiveShadowInterface->enRequestType = REQUEST_MSP_FADER;
      log(&ControllerPlugin,DLT_LOG_INFO,"vChangeSinkSoundProperty_Fader Called for MSP_SOUND_FADER");
      mClientHandlerFCAudioMgr->vSetSinkSoundProperty_Fader(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_ABS,midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS,set.mainSoundProperty.value);
   }
   else
   {
      set.mainSoundProperty.type = MSP_SOUND_FADER;
      mControlReceiveShadowInterface->enRequestType = REQUEST_MSP_FADER;
      if(set.mainSoundProperty.value > 0){
         log(&ControllerPlugin,DLT_LOG_INFO,"vChangeSinkSoundProperty_Fader Called for MSP_SOUND_FADER_RELATIVE with increament");
         mClientHandlerFCAudioMgr->vSetSinkSoundProperty_Fader(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_INC,midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS,set.mainSoundProperty.value);
      }
      else{
         log(&ControllerPlugin,DLT_LOG_INFO,"vChangeSinkSoundProperty_Fader Called for MSP_SOUND_FADER_RELATIVE with decreament by:",abs(set.mainSoundProperty.value));
         mClientHandlerFCAudioMgr->vSetSinkSoundProperty_Fader(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_DEC,midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS,abs(set.mainSoundProperty.value));
      }
   }
}

/************************************************************************
* FUNCTION        : hookUserSetMainSinkSoundProperty
* DESCRIPTION     : To set main sink sound property
* PARAMETER       : const am_sinkID_t, const am_MainSoundProperty_s
* RETURNVALUE     :  am_Error_e
* HISTORY         : GENIVI
************************************************************************/
am_Error_e CAmControlSenderBase::hookUserSetMainSinkSoundProperty(const am_sinkID_t sinkID, const am_MainSoundProperty_s & soundProperty)
{
   log(&ControllerPlugin,DLT_LOG_INFO,"Entered hookUserSetMainSinkSoundProperty");

   //Check for pointer
   if((NULL == mClientHandlerFCAudioMgr) || (NULL == mControlReceiveShadowInterface) || (sinkID == 0))
      return E_UNKNOWN;

   rMainSinkSoundPropertySet set;
   set.sinkID = sinkID;
   set.mainSoundProperty = soundProperty;

   //Its User Request
   mControlReceiveShadowInterface->m_bUserRequest = true;

   //check type of sound property
   switch(soundProperty.type)
   {
       //Bass
      case MSP_SOUND_BASS :
      case MSP_SOUND_BASS_RELATIVE :
      {
         vChangeSinkSoundProperty_Bass(set,soundProperty.type);
         break;
      }
      //Treble
      case MSP_SOUND_TREBLE :
      case MSP_SOUND_TREBLE_RELATIVE :
      {
         vChangeSinkSoundProperty_Treble(set,soundProperty.type);
         break;
      }
      //Middle
      case MSP_SOUND_MIDDLE :
      case MSP_SOUND_MIDDLE_RELATIVE :
      {
         vChangeSinkSoundProperty_MidRange(set,soundProperty.type);
         break;
      }
      //Loudness
      case MSP_SOUND_LOUDNESS :
      case MSP_SOUND_LOUDNESS_RELATIVE :
      {
         vChangeSinkSoundProperty_Loudness(set,soundProperty.type);
         break;
      }
      //Balance
      case MSP_SOUND_BALANCE :
      case MSP_SOUND_BALANCE_RELATIVE :
      {
         vChangeSinkSoundProperty_Balance(set,soundProperty.type);
         break;
      }
      //Fader
      case MSP_SOUND_FADER :
      case MSP_SOUND_FADER_RELATIVE :
      {
         vChangeSinkSoundProperty_Fader(set,soundProperty.type);
         break;
      }
      //Dynamic Compression
      case MSP_SOUND_DYN_COMPRESSION :
      {
         mControlReceiveShadowInterface->enRequestType = REQUEST_MSP_DYNA_COMPRESSION;
         log(&ControllerPlugin,DLT_LOG_INFO,"hookUserSetMainSourceSoundProperty Called for MSP_SOUND_DYN_COMPRESSION");
         mClientHandlerFCAudioMgr->vSetSinkSoundProperty_DynCompression(soundProperty.value,midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS);
         break;
      }
      //Equalizer Preset
      case MSP_EQUALIZER_PRESET :
      {
         mControlReceiveShadowInterface->enRequestType = REQUEST_MSP_EQU_PRESET;
         log(&ControllerPlugin,DLT_LOG_INFO,"hookUserSetMainSourceSoundProperty Called for MSP_EQUALIZER_PRESET");
         mClientHandlerFCAudioMgr->vSetSinkSoundProperty_PresetEquilizer(soundProperty.value,midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS);
         break;
      }
      //SDVC
      case MSP_SPEED_DEPENDENT_VOLUME :
      {
         mControlReceiveShadowInterface->enRequestType = REQUEST_MSP_SDVC;
         log(&ControllerPlugin,DLT_LOG_INFO,"hookUserSetMainSourceSoundProperty Called for MSP_SPEED_DEPENDENT_VOLUME :",soundProperty.value);
         mClientHandlerFCAudioMgr->vSetSinkSoundProperty_SpeedDepVolChange(soundProperty.value,midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS);
         break;
      }
      //Sound optimization
      case MSP_SOUND_OPTIMIZATION :
      {
         mControlReceiveShadowInterface->enRequestType = REQUEST_MSP_SND_OPTIMIZATION;
         log(&ControllerPlugin,DLT_LOG_INFO,"hookUserSetMainSourceSoundProperty Called for MSP_SOUND_OPTIMIZATION");
         mClientHandlerFCAudioMgr->vSetSinkSoundProperty_SoundOptimization(soundProperty.value,midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS);
         break;
      }
      //Beep
      case MSP_LEVEL_BEEP :
      {
         mControlReceiveShadowInterface->enRequestType = REQUEST_MSP_BEEP_LEVEL;
         log(&ControllerPlugin,DLT_LOG_INFO,"hookUserSetMainSourceSoundProperty Called for MSP_LEVEL_BEEP");
         mClientHandlerFCAudioMgr->vSetSystemSoundProperty_Beep(soundProperty.value,midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS);
         break;
      }
      //default
      default :
      {
         log(&ControllerPlugin,DLT_LOG_INFO,"hookUserSetMainSourceSoundProperty Called for unknown...");
         break;
      }
   }

   //push in List
   mListMainSinkSoundPropertyChanges.push_back(set);
   return E_OK;
}

/************************************************************************
* FUNCTION        : hookUserSetMainSourceSoundProperty
* DESCRIPTION     : To set main source sound property
* PARAMETER       : const am_sinkID_t, const am_MainSoundProperty_s
* RETURNVALUE     :  am_Error_e
* HISTORY         : GENIVI
************************************************************************/
am_Error_e CAmControlSenderBase::hookUserSetMainSourceSoundProperty(const am_sourceID_t sourceID, const am_MainSoundProperty_s & soundProperty)
{
   log(&ControllerPlugin,DLT_LOG_INFO,"Entered hookUserSetMainSourceSoundProperty");

   if((NULL == mClientHandlerFCAudioMgr) || (NULL == mControlReceiveShadowInterface) || (sourceID == 0))
      return E_NON_EXISTENT;

   rMainSourceSoundPropertySet set;
   set.sourceID = sourceID;
   set.mainSoundProperty = soundProperty;

   mControlReceiveShadowInterface->m_bUserRequest = true;

   switch(soundProperty.type)
   {
      case MSP_LEVEL_AUX_IN :
      {
         mControlReceiveShadowInterface->enRequestType = REQUEST_MSP_AUX;
         log(&ControllerPlugin,DLT_LOG_INFO,"hookUserSetMainSourceSoundProperty Called for MSP_LEVEL_AUX_IN");
         mClientHandlerFCAudioMgr->vSetSourceSoundProperty_AuxInLevel(soundProperty.value);
         break;
      }
      default :
      {
         log(&ControllerPlugin,DLT_LOG_INFO,"hookUserSetMainSourceSoundProperty Called for unknown...");
         break;
      }
   }
   mListMainSourceSoundPropertyChanges.push_back(set);
   return E_OK;
}

/************************************************************************
* FUNCTION        : hookUserSetSystemProperty
* DESCRIPTION     : To set System property
* PARAMETER       : const am_SystemProperty_s
* RETURNVALUE     : am_Error_e
* HISTORY         : GENIVI
************************************************************************/
am_Error_e CAmControlSenderBase::hookUserSetSystemProperty(const am_SystemProperty_s & property)
{
   log(&ControllerPlugin,DLT_LOG_INFO,"Entered hookUserSetSystemProperty,type:",property.type,",value:",property.value);

   if((NULL == mClientHandlerFCAudioMgr) || (NULL == mControlReceiveShadowInterface))
      return E_NON_EXISTENT;

   mControlReceiveShadowInterface->m_bUserRequest = true;

   switch(property.type)
   {
      case SYP_BEEP :
      {
         mControlReceiveShadowInterface->enRequestType = REQUEST_SYP_BEEP;
         log(&ControllerPlugin,DLT_LOG_INFO,"hookUserSetSystemProperty Called for SYP_BEEP");
         mClientHandlerFCAudioMgr->vSetSystemSoundProperty_Beep(property.value,midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS);
         break;
      }
      case SYP_FADER_STATUS_ACTIVE :
      {
         mControlReceiveShadowInterface->enRequestType = REQUEST_SYP_FADER_STATUS_ACTIVE;
         log(&ControllerPlugin,DLT_LOG_INFO,"hookUserSetSystemProperty Called for SYP_FADER_STATUS_ACTIVE");
         //mClientHandlerFCAudioMgr->vSetSourceSoundProperty_Beep(midw_fi_tcl_e8_AudioSoundType::FI_EN_AUDIO_SND_TYPE_ABS,midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS,soundProperty.value);
         break;
      }
      case SYP_VOLMODE_STREAM_MAIN_1 :
      case SYP_VOLMODE_STREAM_MIX_1 :
      {
         mControlReceiveShadowInterface->enRequestType = REQUEST_SYP_VOL_MODE;
         log(&ControllerPlugin,DLT_LOG_INFO,"hookUserSetSystemProperty Called for SYP_VOLMODE_STREAM_MAIN_1");

         uint8_t u8VolMode = (uint8_t)property.value;
         mClientHandlerFCAudioMgr->vSetSystemSoundProperty_VolumeMode(u8VolMode,midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS);
         break;
      }
   }

   mListMainSystemSoundPropertyChanges.push_back(property);
   return E_OK;
}

/************************************************************************
* FUNCTION        : hookUserVolumeChange
* DESCRIPTION     : to set user volume
* PARAMETER       : const am_sinkID_t, const am_mainVolume_t
* RETURNVALUE     : am_Error_e
* HISTORY         :
*     04.09.12 Vyankatesh VD  Initial Revision.
************************************************************************/
am_Error_e CAmControlSenderBase::hookUserVolumeChange(const am_sinkID_t SinkID, const am_mainVolume_t newVolume)
{
   log(&ControllerPlugin,DLT_LOG_INFO,"hookUserVolumeChange called with sinkID=", SinkID, "newVolume=", newVolume);

    assert(SinkID!=0);

    if(mControlReceiveShadowInterface){
      mControlReceiveShadowInterface->m_bUserRequest = true;
      mControlReceiveShadowInterface->enRequestType = REQUEST_VOLUME;
    }

   if(mClientHandlerFCAudioMgr)
   {
      log(&ControllerPlugin,DLT_LOG_INFO,"hookUserVolumeChange, Calling vSetVolume()");
      mClientHandlerFCAudioMgr->vSetVolume(midw_fi_tcl_e8_AudioVolumeType::FI_EN_AUDIO_VAL_TYPE_ABS,
                                 midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS,newVolume);
   }

   log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase: After calling setSinkVolume(),successfully returned");
   return E_OK;
}

/************************************************************************
* FUNCTION     : hookUserVolumeStep
* DESCRIPTION  : To set volume in step
* PARAMETER    : am_sinkID_t,int16_t
* RETURNVALUE  : am_Error_e
* HISTORY      :
*     04.09.12 Vyankatesh VD  Initial Revision.
************************************************************************/
am_Error_e CAmControlSenderBase::hookUserVolumeStep(const am_sinkID_t SinkID, const int16_t i16Step)
{
   logInfo("CAmControlSenderBase: hookUserVolumeStep called with sinkID=", SinkID, "increment=", i16Step);
   assert(SinkID!=0);
   rMainVolumeSet set;
   set.sinkID = SinkID;
   am_Sink_s sink;

   if(mControlReceiveShadowInterface){
      mControlReceiveShadowInterface->m_bUserRequest = true;
      mControlReceiveShadowInterface->enRequestType = REQUEST_VOLUME;
   }

   if(mClientHandlerFCAudioMgr)
   {
      log(&ControllerPlugin,DLT_LOG_INFO,"hookUserVolumeStep, Calling vSetVolume()");

      if(i16Step > 0)
      {
         am_Sink_s sinkData;
         if(mControlReceiveInterface)
         {
            if(clGeniviAudioCtrlAdapter::bIsCurrentActiveAudiosource(SourceID(AudioSources::MUTE_SYSTEM,0)))
            {
              log(&ControllerPlugin,DLT_LOG_INFO,"hookUserVolumeStep,Already MUTED, Calling FcAudioMgr->vSetVolume(FI_EN_AUDIO_VAL_TYPE_ABS) value: ",i16Step);
                mClientHandlerFCAudioMgr->vSetVolume(midw_fi_tcl_e8_AudioVolumeType::FI_EN_AUDIO_VAL_TYPE_ABS,
                                        midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS,i16Step);
            }
            else
            {
              log(&ControllerPlugin,DLT_LOG_INFO,"hookUserVolumeStep, Calling FcAudioMgr->vSetVolume(FI_EN_AUDIO_VAL_TYPE_INC) by ",i16Step);
              mClientHandlerFCAudioMgr->vSetVolume(midw_fi_tcl_e8_AudioVolumeType::FI_EN_AUDIO_VAL_TYPE_INC,
                                               midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS,i16Step);
            }
         }
      }
      else
      {
         log(&ControllerPlugin,DLT_LOG_INFO,"hookUserVolumeStep, Calling FcAudioMgr->vSetVolume(FI_EN_AUDIO_VAL_TYPE_DEC) by ",i16Step);
         mClientHandlerFCAudioMgr->vSetVolume(midw_fi_tcl_e8_AudioVolumeType::FI_EN_AUDIO_VAL_TYPE_DEC,
                                 midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS,abs(i16Step));
      }
      return E_OK;
   }
   return E_UNKNOWN;
}

/************************************************************************
* FUNCTION     : hookUserSetSinkMuteState
* DESCRIPTION  : to set sink in MUTE/DEMUTE state
* PARAMETER    : am_sinkID_t, am_MuteState_e
* RETURNVALUE  : am_Error_e
* HISTORY      :
*     04.09.12 Vyankatesh VD  Initial Revision.
************************************************************************/
am_Error_e CAmControlSenderBase::hookUserSetSinkMuteState(const am_sinkID_t sinkID, const am_MuteState_e muteState)
{
    assert(sinkID!=0);
    rMainVolumeSet set;
    set.sinkID = sinkID;
    set.muteState = muteState;

    if(mControlReceiveShadowInterface)
    {
      mControlReceiveShadowInterface->m_bUserRequest = true;
      mControlReceiveShadowInterface->enRequestType = REQUEST_MUTE_DEMUTE;
    }

    log(&ControllerPlugin,DLT_LOG_INFO,"hookUserSetSinkMuteState called with sinkID=", sinkID, "muteState=", muteState);

  mListOpenMuteDemuteChanges.push_back(set);

    if(MS_MUTED == muteState )
    {
      //m_bUserRequest_SinkMuted = true;
      log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::hookUserSetSinkMuteState, Its MUTE request");

     return clGeniviAudioCtrlAdapter::RequestSourceOn(AudioSources::MUTE_ENTERTAIN);
    }
    else if(MS_UNMUTED == muteState)
    {
       //m_bUserRequest_SinkMuted = false;
       log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::hookUserSetSinkMuteState, Its DEMUTE request");
       if(mClientHandlerFCAudioMgr)
         mClientHandlerFCAudioMgr->vPerformOnVolLimit();
       return clGeniviAudioCtrlAdapter::RequestSourceOff(AudioSources::MUTE_ENTERTAIN);
    }
    else
    {
      log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::hookUserSetSinkMuteState , undefined muteState");
      return E_OUT_OF_RANGE;
    }
}
/************************************************************************
* FUNCTION        : hookSystemRegisterDomain
* DESCRIPTION     : to register domain
* PARAMETER       : const am_Domain_s, am_domainID_t
* RETURNVALUE     :  am_Error_e
* HISTORY         : GENIVI
************************************************************************/
am_Error_e CAmControlSenderBase::hookSystemRegisterDomain(const am_Domain_s & domainData, am_domainID_t & domainID)
{
    am_Error_e error = E_UNKNOWN;
   //Enter domain data in database
    if(mControlReceiveInterface)
      error = mControlReceiveInterface->enterDomainDB(domainData, domainID);
    logInfo("enterDomainDB returns domainID = %d", domainID);
    clGeniviAudioCtrlAdapter::AddDomainID(domainID);
    logInfo("CAmControlSenderBase::hookSystemRegisterDomain: exit");
   return error;
}

/************************************************************************
* FUNCTION        : hookSystemDeregisterDomain
* DESCRIPTION     : to deregister domain
* PARAMETER       : const am_domainID_t
* RETURNVALUE     :  am_Error_e
* HISTORY         : GENIVI
************************************************************************/
am_Error_e CAmControlSenderBase::hookSystemDeregisterDomain(const am_domainID_t domainID)
{
    am_Error_e error = E_UNKNOWN;
    //Remove domain data from database
    if(mControlReceiveInterface)
      error = mControlReceiveInterface->removeDomainDB(domainID);

   return error;
}

/************************************************************************
* FUNCTION        : hookSystemDomainRegistrationComplete
* DESCRIPTION     : to indicate system dmoain registration complete
* PARAMETER       : const am_domainID_t
* RETURNVALUE     :  void
* HISTORY         : GENIVI
************************************************************************/
void CAmControlSenderBase::hookSystemDomainRegistrationComplete(const am_domainID_t domainID)
{
    (void) domainID;
}

/************************************************************************
* FUNCTION        : hookSystemRegisterSink
* DESCRIPTION     : to register sink
* PARAMETER       : const am_Sink_s , am_sinkID_t
* RETURNVALUE     :  am_Error_e
* HISTORY         : GENIVI
************************************************************************/
am_Error_e CAmControlSenderBase::hookSystemRegisterSink(const am_Sink_s & sinkData, am_sinkID_t & sinkID)
{
   log(&ControllerPlugin,DLT_LOG_INFO,"hookSystemRegisterSink called with sinkID=", sinkID);
    am_Error_e error = E_UNKNOWN;
   //this application does not do anything with it -> but some product might want to take influence here
    if(mControlReceiveInterface)
      error = mControlReceiveInterface->enterSinkDB(sinkData, sinkID);

   return error;
}

/************************************************************************
* FUNCTION        : hookSystemDeregisterSink
* DESCRIPTION     : to deregister sink
* PARAMETER       : const am_sinkID_t
* RETURNVALUE     :  am_Error_e
* HISTORY         : GENIVI
************************************************************************/
am_Error_e CAmControlSenderBase::hookSystemDeregisterSink(const am_sinkID_t sinkID)
{
   log(&ControllerPlugin,DLT_LOG_INFO,"hookSystemDeregisterSink called with sinkID=", sinkID);
    am_Error_e error = E_UNKNOWN;
   //this application does not do anything with it -> but some product might want to take influence here
    if(mControlReceiveInterface)
      error = mControlReceiveInterface->removeSinkDB(sinkID);

   return error;
}
/************************************************************************
* FUNCTION        : hookSystemRegisterSource
* DESCRIPTION     : to register source
* PARAMETER       : const am_Source_s, am_sourceID_t
* RETURNVALUE     :  am_Error_e
* HISTORY         : GENIVI
************************************************************************/
am_Error_e CAmControlSenderBase::hookSystemRegisterSource(const am_Source_s & sourceData, am_sourceID_t & sourceID)
{
   log(&ControllerPlugin,DLT_LOG_INFO,"hookSystemRegisterSource called with sourceID=", sourceID);
    am_Error_e error = E_UNKNOWN;
   //this application does not do anything with it -> but some product might want to take influence here
    if(mControlReceiveInterface)
      error = mControlReceiveInterface->enterSourceDB(sourceData, sourceID);

   return error;
}

/************************************************************************
* FUNCTION        : hookSystemDeregisterSource
* DESCRIPTION     : to deregister source
* PARAMETER       : const am_sourceID_t
* RETURNVALUE     :  am_Error_e
* HISTORY         : GENIVI
************************************************************************/
am_Error_e CAmControlSenderBase::hookSystemDeregisterSource(const am_sourceID_t sourceID)
{
   log(&ControllerPlugin,DLT_LOG_INFO,"hookSystemDeregisterSource called with sourceID=", sourceID);
    am_Error_e error = E_UNKNOWN;
   //this application does not do anything with it -> but some product might want to take influence here
    if(mControlReceiveInterface)
      error = mControlReceiveInterface->removeSourceDB(sourceID);

   return error;
}
/************************************************************************
* FUNCTION        : hookSystemRegisterGateway
* DESCRIPTION     : to register gateway
* PARAMETER       : const am_Gateway_s , am_gatewayID_t
* RETURNVALUE     :  am_Error_e
* HISTORY         : GENIVI
************************************************************************/
am_Error_e CAmControlSenderBase::hookSystemRegisterGateway(const am_Gateway_s & gatewayData, am_gatewayID_t & gatewayID)
{
    am_Error_e error = E_UNKNOWN;
    //this application does not do anything with it -> but some product might want to take influence here
    if(mControlReceiveInterface)
      error = mControlReceiveInterface->enterGatewayDB(gatewayData, gatewayID);

    return error;
}

/************************************************************************
* FUNCTION        : hookSystemDeregisterGateway
* DESCRIPTION     : to deregister gateway
* PARAMETER       : const  am_gatewayID_t
* RETURNVALUE     :  am_Error_e
* HISTORY         : GENIVI
************************************************************************/
am_Error_e CAmControlSenderBase::hookSystemDeregisterGateway(const am_gatewayID_t gatewayID)
{
    am_Error_e error = E_UNKNOWN;
   //this application does not do anything with it -> but some product might want to take influence here
    if(mControlReceiveInterface)
      error = mControlReceiveInterface->removeGatewayDB(gatewayID);

   return error;
}

/************************************************************************
* FUNCTION        : hookSystemRegisterCrossfader
* DESCRIPTION     : to register Crossfader
* PARAMETER       : const  am_Crossfader_s, am_crossfaderID_t
* RETURNVALUE     :  am_Error_e
* HISTORY         : GENIVI
************************************************************************/
am_Error_e CAmControlSenderBase::hookSystemRegisterCrossfader(const am_Crossfader_s & crossfaderData, am_crossfaderID_t & crossfaderID)
{
    am_Error_e error = E_UNKNOWN;
   //this application does not do anything with it -> but some product might want to take influence here
    if(mControlReceiveInterface)
      error = mControlReceiveInterface->enterCrossfaderDB(crossfaderData, crossfaderID);

   return error;
}

/************************************************************************
* FUNCTION        : hookSystemDeregisterCrossfader
* DESCRIPTION     : to deregister Crossfader
* PARAMETER       : const  am_crossfaderID_t
* RETURNVALUE     :  am_Error_e
* HISTORY         : GENIVI
************************************************************************/
am_Error_e CAmControlSenderBase::hookSystemDeregisterCrossfader(const am_crossfaderID_t crossfaderID)
{
    am_Error_e error = E_UNKNOWN;
   //this application does not do anything with it -> but some product might want to take influence here
    if(mControlReceiveInterface)
      error = mControlReceiveInterface->removeCrossfaderDB(crossfaderID);

   return error;
}

/************************************************************************
* FUNCTION        : hookSystemSinkVolumeTick
* DESCRIPTION     : used to indicate volumechanges during a ramp
* PARAMETER       : const  am_Handle_s, const am_sinkID_t, const am_volume_t
* RETURNVALUE     : void
* HISTORY         : GENIVI
************************************************************************/
void CAmControlSenderBase::hookSystemSinkVolumeTick(const am_Handle_s handle, const am_sinkID_t sinkID, const am_volume_t volume)
{
    (void) handle;
    (void) sinkID;
    (void) volume;
}

/************************************************************************
* FUNCTION        : hookSystemSourceVolumeTick
* DESCRIPTION     : used to indicate volumechanges during a ramp
* PARAMETER       : const  am_Handle_s, const am_sinkID_t, const am_volume_t
* RETURNVALUE     : void
* HISTORY         : GENIVI
************************************************************************/
void CAmControlSenderBase::hookSystemSourceVolumeTick(const am_Handle_s handle, const am_sourceID_t sourceID, const am_volume_t volume)
{
    (void) handle;
    (void) sourceID;
    (void) volume;
}

/************************************************************************
* FUNCTION        : hookSystemInterruptStateChange
* DESCRIPTION     : is called when an low level interrupt changed its state
* PARAMETER       : const am_sourceID_t, const am_InterruptState_e
* RETURNVALUE     : void
* HISTORY         : GENIVI
************************************************************************/
void CAmControlSenderBase::hookSystemInterruptStateChange(const am_sourceID_t sourceID, const am_InterruptState_e interruptState)
{
    (void) sourceID;
    (void) interruptState;
}

/************************************************************************
* FUNCTION        : hookSystemSinkAvailablityStateChange
* DESCRIPTION     : called when a sink changed its availability
* PARAMETER       : const am_sinkID_t, const am_Availability_s
* RETURNVALUE     : void
* HISTORY         : GENIVI
************************************************************************/
void CAmControlSenderBase::hookSystemSinkAvailablityStateChange(const am_sinkID_t sinkID, const am_Availability_s & availability)
{
    (void) sinkID;
    (void) availability;
}

/************************************************************************
* FUNCTION        : hookSystemSourceAvailablityStateChange
* DESCRIPTION     : called when a source changed its availability
* PARAMETER       : const am_sourceID_t, const am_Availability_s
* RETURNVALUE     : void
* HISTORY         : GENIVI
************************************************************************/
void CAmControlSenderBase::hookSystemSourceAvailablityStateChange(const am_sourceID_t sourceID, const am_Availability_s & availability)
{
    (void) sourceID;
    (void) availability;
}

/************************************************************************
* FUNCTION        : hookSystemDomainStateChange
* DESCRIPTION     : called when domainstate was changed
* PARAMETER       : const am_domainID_t, const am_DomainState_e
* RETURNVALUE     : void
* HISTORY         : GENIVI
************************************************************************/
void CAmControlSenderBase::hookSystemDomainStateChange(const am_domainID_t domainID, const am_DomainState_e state)
{
    (void) domainID;
    (void) state;
}

/************************************************************************
* FUNCTION        : hookSystemReceiveEarlyData
* DESCRIPTION     : called when early data was received
* PARAMETER       : const std::vector<am_EarlyData_s>
* RETURNVALUE     : void
* HISTORY         : GENIVI
************************************************************************/
void CAmControlSenderBase::hookSystemReceiveEarlyData(const std::vector<am_EarlyData_s> & data)
{
    (void) data;
}

/************************************************************************
* FUNCTION        : hookSystemSpeedChange
* DESCRIPTION     : this hook provides information about speed changes.
* PARAMETER       : const am_speed_t
* RETURNVALUE     : void
* HISTORY         : GENIVI
************************************************************************/
void CAmControlSenderBase::hookSystemSpeedChange(const am_speed_t speed)
{
    (void) speed;
}

/************************************************************************
* FUNCTION        : hookSystemTimingInformationChanged
* DESCRIPTION     : this hook is fired whenever the timing information of a mainconnection has changed.
* PARAMETER       : const am_mainConnectionID_t , const am_timeSync_t
* RETURNVALUE     : void
* HISTORY         : GENIVI
************************************************************************/
void CAmControlSenderBase::hookSystemTimingInformationChanged(const am_mainConnectionID_t mainConnectionID, const am_timeSync_t time1)
{
    (void) mainConnectionID;
    (void) time1;
}

/************************************************************************
* FUNCTION        : cbAckConnect
* DESCRIPTION     : ack for connect
* PARAMETER       : const am_Handle_s , const am_Error_e
* RETURNVALUE     : void
* HISTORY         : GENIVI
************************************************************************/
void CAmControlSenderBase::cbAckConnect(const am_Handle_s handle, const am_Error_e errorID)
{
   clGeniviAudioCtrlAdapter::cbAckConnect(handle, errorID);
}

/************************************************************************
* FUNCTION        : cbAckDisconnect
* DESCRIPTION     : ack for disconnect
* PARAMETER       : const am_Handle_s , const am_Error_e
* RETURNVALUE     : void
* HISTORY         : GENIVI
************************************************************************/
void CAmControlSenderBase::cbAckDisconnect(const am_Handle_s handle, const am_Error_e error)
{
   clGeniviAudioCtrlAdapter::cbAckDisconnect(handle, error);
}

/************************************************************************
* FUNCTION        : cbAckCrossFade
* DESCRIPTION     : ack for crossfade
* PARAMETER       : const am_Handle_s , const am_HotSink_e , const am_Error_e
* RETURNVALUE     : void
* HISTORY         : GENIVI
************************************************************************/
void CAmControlSenderBase::cbAckCrossFade(const am_Handle_s handle, const am_HotSink_e hostsink, const am_Error_e error)
{
    (void) handle;
    (void) hostsink;
    (void) error;
}

/************************************************************************
* FUNCTION     : cbAckSetSinkVolumeChange
* DESCRIPTION     : ack for SetSinkVolumeChange
* PARAMETER    : const am_Handle_s , const am_volume_t , const am_Error_e
* RETURNVALUE     : void
* HISTORY         :
*     04.09.12 Vyankatesh VD  Initial Revision.
************************************************************************/
void CAmControlSenderBase::cbAckSetSinkVolumeChange(const am_Handle_s handle, const am_volume_t volume, const am_Error_e error)
{
   if(E_OK == error)
    {
      log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::cbAckSetSinkVolumeChange successfully called, volume :",volume);

        std::vector<rMainVolumeSet>::iterator it(mListOpenVolumeChanges.begin());
        for (; it != mListOpenVolumeChanges.end(); ++it)
        {
            if (handle.handle == it->handle.handle)
            {
               log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::cbAckSetSinkVolumeChange,Sending ack to Command Plugin, volume :",volume);
                if(mControlReceiveInterface)
               mControlReceiveInterface->changeSinkMainVolumeDB(volume, it->sinkID);
                mListOpenVolumeChanges.erase(it);
              break;
            }
        }
    }

}

/************************************************************************
* FUNCTION     : cbAckSetSourceVolumeChange
* DESCRIPTION     : ack for SetSourceVolumeChange
* PARAMETER    : const am_Handle_s , const am_volume_t , const am_Error_e
* RETURNVALUE     : void
* HISTORY         : GENIVI
************************************************************************/
void CAmControlSenderBase::cbAckSetSourceVolumeChange(const am_Handle_s handle, const am_volume_t voulme, const am_Error_e error)
{
    (void) error;
    (void) voulme;
    (void) handle;
}

/************************************************************************
* FUNCTION     : cbAckSetSourceState
* DESCRIPTION     : ack for SetSourceState
* PARAMETER    : const am_Handle_s , const am_Error_e
* RETURNVALUE     : void
* HISTORY         : GENIVI
************************************************************************/
//vdu1kor
void CAmControlSenderBase::cbAckSetSourceState(const am_Handle_s handle, const am_Error_e error)
{
    (void) error;
    (void) handle;
    //TODO Need to call GAM_AckSourceState()
    clGeniviAudioCtrlAdapter::cbAckSourceState(handle, error);

}

/************************************************************************
* FUNCTION     : cbAckSetSourceSoundProperty
* DESCRIPTION     : ack for SetSourceSoundProperty
* PARAMETER    : const am_Handle_s , const am_Error_e
* RETURNVALUE     : void
* HISTORY         : GENIVI
************************************************************************/
void CAmControlSenderBase::cbAckSetSourceSoundProperty(const am_Handle_s handle, const am_Error_e error)
{
   if(E_OK == error)
   {
      std::vector<rMainSourceSoundPropertySet>::iterator it(mListMainSourceSoundPropertyChanges.begin());
      for (; it != mListMainSourceSoundPropertyChanges.end(); ++it)
      {
         if (handle.handle == it->handle.handle)
         {
            if(mControlReceiveInterface)
               mControlReceiveInterface->changeMainSourceSoundPropertyDB(it->mainSoundProperty, it->sourceID);
         }
      }
   }
   else{
      log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::cbAckSetSourceSoundProperty  returns errorID=", error);
   }
}

/************************************************************************
* FUNCTION     : cbAckSetSinkSoundProperty
* DESCRIPTION     : ack for SetSinkSoundProperty
* PARAMETER    : const am_Handle_s , const am_Error_e
* RETURNVALUE     : void
* HISTORY         : GENIVI
************************************************************************/
void CAmControlSenderBase::cbAckSetSinkSoundProperty(const am_Handle_s handle, const am_Error_e error)
{
   if(E_OK == error)
   {
      std::vector<rMainSinkSoundPropertySet>::iterator it(mListMainSinkSoundPropertyChanges.begin());
      for (; it != mListMainSinkSoundPropertyChanges.end(); ++it)
      {
         if (handle.handle == it->handle.handle)
         {
            if(mControlReceiveInterface)
               mControlReceiveInterface->changeMainSinkSoundPropertyDB(it->mainSoundProperty, it->sinkID);

         }
      }
   }
   else{
      logInfo("CAmControlSenderBase::cbAckSetSinkSoundProperty  returns errorID=", error);
   }
}


/************************************************************************
 *FUNCTION     : vResponse_SourceChange
 *DESCRIPTION  : to process SourceChange Status response
 *PARAMETER    : amt_tclServiceData*
 *RETURN VALUE : void
 *HISTORY         :
 *20.09.2013   Rev 1.0     RBEI/ECG4 - Vyankatesh VD  Initial Revision
************************************************************************/
void CAmControlSenderBase::vResponse_SourceChange(amt_tclServiceData* poMessage)
{
   log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::vResponse_SourceChange,received SourceRouteActivity status");
   if(AMT_C_U8_CCAMSG_OPCODE_STATUS == poMessage->u8GetOpCode())
   {
    log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::vResponse_SourceChange,received AMT_C_U8_CCAMSG_OPCODE_STATUS, Calling handleSourceStatus");
   midw_audio_route_mgrfi_tclMsgP_SourceRouteActivityStatus oSrcRouteActStatus;

   //Get the data using Helper function
   if(mClientHandlerAudioRouteMgr)
      mClientHandlerAudioRouteMgr->vCopyDatatoFIMessageWrapper(poMessage,oSrcRouteActStatus);

   log(&ControllerPlugin,DLT_LOG_INFO,"vResponse_SourceChange,source",oSrcRouteActStatus.SourceRouteActivity.Source.enType,
      ",PossibleNextSource:",oSrcRouteActStatus.SourceRouteActivity.PossibleNextSource.enType,
      ",Resource:",oSrcRouteActStatus.SourceRouteActivity.Resource.enType,
      ",Activity:",oSrcRouteActStatus.SourceRouteActivity.Activity.enType);

   clGeniviAudioCtrlAdapter::handleSourceStatus(oSrcRouteActStatus.SourceRouteActivity.Activity.enType,
         oSrcRouteActStatus.SourceRouteActivity.Source.enType,
         0);
   }
   else if(AMT_C_U8_CCAMSG_OPCODE_ERROR == poMessage->u8GetOpCode())
   {

     amt_tclServiceDataError oErrorMsg(poMessage);
       tU16 u16ErrorCode = oErrorMsg.u16GetErrorData();

       midw_fi_tcl_e8_AudSource::tenType SourceError = static_cast<midw_fi_tcl_e8_AudSource::tenType>(u16ErrorCode >> 8);
       log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::vResponse_SourceChange,received AMT_C_U8_CCAMSG_OPCODE_ERROR, Calling vHandleSourceError,u16ErrorCode:",u16ErrorCode,
        ",SourceError:",SourceError);

     //SourceError = static_cast<midw_fi_tcl_e8_AudSource::tenType>(2);
     clGeniviAudioCtrlAdapter::vHandleSourceError(SourceError);
   }
   else
   {
     log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::vResponse_SourceChange,Unknown Opcode, Ignoring..");
   }
}

/************************************************************************
 *FUNCTION     : vResponse_SourceAvailability
 *DESCRIPTION  : to process SourceAvailability Status from ARL(AudioRoutingLibrary)
 *PARAMETER    : amt_tclServiceData*
 *RETURN VALUE : void
 *HISTORY         :
 *28.11.2013   Rev 1.0     R.Volkmer
************************************************************************/
void CAmControlSenderBase::vResponse_SourceAvailability(amt_tclServiceData* poMessage)
{
   log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::vResponse_SourceAvailability,received status from ARL");
   midw_audio_route_mgrfi_tclMsgSourceAvailabilityStatus oSourceAvailableStatus;

   //Get the data using Helper function
   if(mClientHandlerAudioRouteMgr)
      mClientHandlerAudioRouteMgr->vCopyDatatoFIMessageWrapper(poMessage,oSourceAvailableStatus);

   bpstl::vector<midw_fi_tcl_SourceAvailablityItem>::const_iterator SrcAvailIterator;

   if(oSourceAvailableStatus.SourceAvailabilityList.empty())
   {
     log(&ControllerPlugin,DLT_LOG_INFO,"vResponse_SourceAvailability,SourceAvailabilityList is empty");
   }
   else
   {
     for(SrcAvailIterator = oSourceAvailableStatus.SourceAvailabilityList.begin();
           SrcAvailIterator != oSourceAvailableStatus.SourceAvailabilityList.end();
              ++SrcAvailIterator)
     {
       midw_fi_tcl_e8_AudSource::tenType Source             = (*SrcAvailIterator).u8Source.enType;
       tU16 SubSourceID                                     = (*SrcAvailIterator).u16SubSource;
       midw_fi_tcl_e8_SrcAvailability::tenType Availability = (*SrcAvailIterator).e8Availability.enType;
       midw_fi_tcl_e8_SrcAvailabilityReason::tenType Reason = (*SrcAvailIterator).e8Reason.enType;

       log(&ControllerPlugin,DLT_LOG_INFO," SourceClass:",(tU8)Source
             , ",SubSource:",(tU16)SubSourceID
             , ",Availability:",(tU8)Availability
             , ",Reason:",(tU8)Reason);

       am_mainConnectionID_t mainConnectionID;

       AudioSources::enAudioSources enInternalSrcClass =
             clGeniviAudioCtrlAdapter::Ext2Int_SrcClass(Source);

       clGeniviAudioCtrlAdapter::SourceAvailabilityChange(Source,
                                                          SubSourceID,
                                                          Availability,
                                                          Reason);
     }
  }
   //clGeniviAudioCtrlAdapter::handleSourceStatus(oSrcRouteActStatus.SourceRouteActivity.Source.enType,
   //      oSrcRouteActStatus.SourceRouteActivity.Activity.enType);

}

/************************************************************************
 *FUNCTION     : vResponse_AudioRoute
 *DESCRIPTION  : to process RequestAudioRoute Status response from ARL(AudioRoutingLibrary)
 *PARAMETER    : amt_tclServiceData*
 *RETURN VALUE : void
 *HISTORY         :
 *10.10.2013   Rev 1.0     RBEI/ECG4 - Vyankatesh VD  Initial Revision
************************************************************************/
void CAmControlSenderBase::vResponse_AudioRoute(amt_tclServiceData* poMessage)
{
   log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::vResponse_AudioRoute,received RequestAudioRoute status from ARL");
   midw_audio_route_mgrfi_tclMsgRequestAudioRouteStatus oAudioRouteStatus;

   //Get the data using Helper function
   if(mClientHandlerAudioRouteMgr)
      mClientHandlerAudioRouteMgr->vCopyDatatoFIMessageWrapper(poMessage,oAudioRouteStatus);

   bpstl::vector<midw_fi_tcl_RequestAudioRouteItem>::const_iterator ReqRouteIterator;

   if(oAudioRouteStatus.RequestAudioRouteList.empty())
   {
     log(&ControllerPlugin,DLT_LOG_INFO,"vResponse_AudioRoute,RequestAudioRouteList is empty");
   }
   else
   {
     for(ReqRouteIterator = oAudioRouteStatus.RequestAudioRouteList.begin();
             ReqRouteIterator != oAudioRouteStatus.RequestAudioRouteList.end();
              ++ReqRouteIterator)
     {
       midw_fi_tcl_e8_AudSource::tenType Source     = (*ReqRouteIterator).u8Source.enType;
       tU16 SubSourceID                             = (*ReqRouteIterator).u16SubSource;
       midw_fi_tcl_e8_SrcActivity::tenType Activity = (*ReqRouteIterator).e8Activity.enType;
       tU8  Reason                                  = (*ReqRouteIterator).u8Reason;

       log(&ControllerPlugin,DLT_LOG_INFO," SourceClass:",(tU8)Source
             , ",SubSource:",(tU16)SubSourceID
             , ",Activity:",Activity
             , ",Reason:",Reason);

       am_mainConnectionID_t mainConnectionID;

       AudioSources::enAudioSources enInternalSrcClass =
             clGeniviAudioCtrlAdapter::Ext2Int_SrcClass(Source);


       if(AudioSources::NONE != enInternalSrcClass){
          if((*ReqRouteIterator).e8Activity.enType ==  midw_fi_tcl_e8_SrcActivity::FI_EN_OFF)
          {
             am_mainConnectionID_t mainConnID = 0;
             clGeniviAudioCtrlAdapter::RequestSourceOff(SourceID(enInternalSrcClass, SubSourceID));
             //getMainConnectionOfSource(enSrcInternal, mainConnID);
//             if(mainConnID == 0)
//             {
//                log(&ControllerPlugin,DLT_LOG_ERROR,"vResponse_AudioRoute,Could not find main connectionID ");
//                return;
//             }
//             clGeniviAudioCtrlAdapter::hookUserDisconnectionRequest(mainConnID);
          }
          if((*ReqRouteIterator).e8Activity.enType ==  midw_fi_tcl_e8_SrcActivity::FI_EN_ON)
          {
             clGeniviAudioCtrlAdapter::RequestSourceOn(SourceID(enInternalSrcClass, SubSourceID));
             //hookUserConnectionRequest(enSrcInternal,1,mainConnectionID);
          }
          if((*ReqRouteIterator).e8Activity.enType ==  midw_fi_tcl_e8_SrcActivity::FI_EN_PAUSE)
          {
             clGeniviAudioCtrlAdapter::RequestSourcePause(SourceID(enInternalSrcClass, SubSourceID));
          }

       }
       else{
           log(&ControllerPlugin,DLT_LOG_INFO,"Not found internal srcID for midew_fi source ",Source);
       }
     }
  }
   //clGeniviAudioCtrlAdapter::handleSourceStatus(oSrcRouteActStatus.SourceRouteActivity.Source.enType,
   //      oSrcRouteActStatus.SourceRouteActivity.Activity.enType);

}

/************************************************************************
 *FUNCTION     : vResponse_Volume
 *DESCRIPTION  : to process VolumeStatus response
 *PARAMETER    : amt_tclServiceData*
 *RETURN VALUE : void
 *HISTORY         :
 *20.09.2013   Rev 1.0     RBEI/ECG4 - Vyankatesh VD  Initial Revision
************************************************************************/
void CAmControlSenderBase::vResponse_Volume(amt_tclServiceData* poMessage)
{
   log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::vResponse_Volume,received volume status");
   midw_mascffi_tclMsgVolumeStatus oVolumeMethodStatus;

   //Get the data using Helper function
   if(mClientHandlerFCAudioMgr)
      mClientHandlerFCAudioMgr->vCopyDatatoFIMessageWrapper(poMessage,oVolumeMethodStatus);

   log(&ControllerPlugin,DLT_LOG_INFO,"vResponse_Volume,VolumeMode:",oVolumeMethodStatus.VolumeMode.enType,
      ",Resource:",oVolumeMethodStatus.Resource.enType);
   log(&ControllerPlugin,DLT_LOG_INFO,"vResponse_Volume,Channel:",oVolumeMethodStatus.Channel.enType,
      ",Volume:",oVolumeMethodStatus.Volume,",Source:",oVolumeMethodStatus.Source.enType);

   log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::processCCAResponse,Sending ack to Command Plugin, volume :",oVolumeMethodStatus.Volume);
   if(mControlReceiveInterface)
      mControlReceiveInterface->changeSinkMainVolumeDB(oVolumeMethodStatus.Volume, 1);

   if(0 == oVolumeMethodStatus.Volume)
   {
     log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::vResponse_Volume,VolumeSet == 0,Calling clGeniviAudioCtrlAdapter::RequestSourceOn[ENTERTAINMENT_MUTE]");
     clGeniviAudioCtrlAdapter::RequestSourceOn(SourceID(AudioSources::MUTE_ENTERTAIN,0));
   }else if(oVolumeMethodStatus.Volume > 0) //TODO : Need to check
   {
   if(true == clGeniviAudioCtrlAdapter::bIsCurrentActiveAudiosource(SourceID(AudioSources::MUTE_ENTERTAIN,0)))
   {
     log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::vResponse_Volume,VolumeSet > 0,Calling clGeniviAudioCtrlAdapter::RequestSourceOff[ENTERTAINMENT_MUTE]");
     clGeniviAudioCtrlAdapter::RequestSourceOff(SourceID(AudioSources::MUTE_ENTERTAIN, 0));
     }
   if(true == clGeniviAudioCtrlAdapter::bIsCurrentActiveAudiosource(SourceID(AudioSources::MUTE_SYSTEM,0)))
   {
       log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::vResponse_Volume,VolumeSet > 0,Calling clGeniviAudioCtrlAdapter::RequestSourceOff[SYSTEM_MUTE]");
       clGeniviAudioCtrlAdapter::RequestSourceOff(SourceID(AudioSources::MUTE_SYSTEM, 0));
     }
   }
}

/************************************************************************
 *FUNCTION     : vResponse_Mute
 *DESCRIPTION  : to process MuteStatus response
 *PARAMETER    : amt_tclServiceData*
 *RETURN VALUE : void
 *HISTORY         :
 *20.09.2013   Rev 1.0     RBEI/ECG4 - Vyankatesh VD  Initial Revision
************************************************************************/
void CAmControlSenderBase::vResponse_Mute(amt_tclServiceData* poMessage)
{
    (void) poMessage;
    log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::vResponse_Mute,received Mute Method Result");

     //special Handling for UserMute request, response is send back to HMI
     if(!mListOpenMuteDemuteChanges.empty())
     {
       std::vector<rMainVolumeSet>::iterator it(mListOpenMuteDemuteChanges.begin());
       for (; it != mListOpenMuteDemuteChanges.end(); ++it)
       {
         log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::vResponse_Mute,calling changeSinkMuteStateDB,muteState:",it->muteState);
         if(mControlReceiveInterface)
            mControlReceiveInterface->changeSinkMuteStateDB(it->muteState, it->sinkID);
         mListOpenMuteDemuteChanges.erase(it);
         break;
       }
     }
     else{
       log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::vResponse_Mute,mListOpenMuteDemuteChanges is empty.");
     }
}

/************************************************************************
 *FUNCTION     : vResponse_SinkSndProp_Bass
 *DESCRIPTION  : to process BassStatus response
 *PARAMETER    : amt_tclServiceData*
 *RETURN VALUE : void
 *HISTORY         :
 *20.09.2013   Rev 1.0     RBEI/ECG4 - Vyankatesh VD  Initial Revision
************************************************************************/
void CAmControlSenderBase::vResponse_SinkSndProp_Bass(amt_tclServiceData* poMessage)
{
   log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::vResponse_SinkSndProp_Bass,received bass status");
   midw_mascffi_tclMsgBassStatus oBassStatus;

   //Get the data using Helper function
   if(mClientHandlerFCAudioMgr)
      mClientHandlerFCAudioMgr->vCopyDatatoFIMessageWrapper(poMessage,oBassStatus);
   log(&ControllerPlugin,DLT_LOG_INFO,"vResponse_SinkSndProp_Bass,Bass:",oBassStatus.Bass,",Resource:",
         oBassStatus.Resource.enType,",Channel:",oBassStatus.Channel.enType);

   if(!mListMainSinkSoundPropertyChanges.empty())
   {
      std::vector<rMainSinkSoundPropertySet>::iterator it(mListMainSinkSoundPropertyChanges.begin());
      for (; it != mListMainSinkSoundPropertyChanges.end(); ++it)
      {
         log(&ControllerPlugin,DLT_LOG_INFO,"vResponse_SinkSndProp_Bass,received bass status,calling changeMainSinkSoundPropertyDB,it->mainSoundProperty.type:",it->mainSoundProperty.type,",value:",it->mainSoundProperty.value);

         it->mainSoundProperty.value = oBassStatus.Bass;
         if(mControlReceiveInterface)
            mControlReceiveInterface->changeMainSinkSoundPropertyDB(it->mainSoundProperty, it->sinkID);
         mListMainSinkSoundPropertyChanges.erase(it);
         break;
      }
   }
   else{
      log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::processCCAResponse,received bass status,ignoring bass status....");
   }
}

/************************************************************************
 *FUNCTION     : vResponse_SinkSndProp_Treble
 *DESCRIPTION  : to process TrebleStatus response
 *PARAMETER    : amt_tclServiceData*
 *RETURN VALUE : void
 *HISTORY         :
 *20.09.2013   Rev 1.0     RBEI/ECG4 - Vyankatesh VD  Initial Revision
************************************************************************/
void CAmControlSenderBase::vResponse_SinkSndProp_Treble(amt_tclServiceData* poMessage)
{
   log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::vResponse_SinkSndProp_Treble,received treble status");
   midw_mascffi_tclMsgTrebleStatus oTrebleStatus;

   //Get the data using Helper function
   if(mClientHandlerFCAudioMgr)
      mClientHandlerFCAudioMgr->vCopyDatatoFIMessageWrapper(poMessage,oTrebleStatus);

   log(&ControllerPlugin,DLT_LOG_INFO,"vResponse_SinkSndProp_Treble,Treble:",oTrebleStatus.Treble,",Resource:",
      oTrebleStatus.Resource.enType,",Channel:",oTrebleStatus.Channel.enType);

   if(!mListMainSinkSoundPropertyChanges.empty())
   {
      std::vector<rMainSinkSoundPropertySet>::iterator it(mListMainSinkSoundPropertyChanges.begin());
      for (; it != mListMainSinkSoundPropertyChanges.end(); ++it)
      {
         log(&ControllerPlugin,DLT_LOG_INFO,"vResponse_SinkSndProp_Treble,received Treble status,calling changeMainSinkSoundPropertyDB,type:",it->mainSoundProperty.type,",value:",it->mainSoundProperty.value);

         it->mainSoundProperty.value = oTrebleStatus.Treble;
         if(mControlReceiveInterface)
            mControlReceiveInterface->changeMainSinkSoundPropertyDB(it->mainSoundProperty, it->sinkID);
         mListMainSinkSoundPropertyChanges.erase(it);
         break;
      }
   }
   else{
      log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::processCCAResponse,received Treble status,ignoring Treble status....");
   }
}

/************************************************************************
 *FUNCTION     : vResponse_SinkSndProp_MidRange
 *DESCRIPTION  : to process MidRangeStatus response
 *PARAMETER    : amt_tclServiceData*
 *RETURN VALUE : void
 *HISTORY         :
 *20.09.2013   Rev 1.0     RBEI/ECG4 - Vyankatesh VD  Initial Revision
************************************************************************/
void CAmControlSenderBase::vResponse_SinkSndProp_MidRange(amt_tclServiceData* poMessage)
{
   log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::vResponse_SinkSndProp_MidRange,received Midrange status");
   midw_mascffi_tclMsgMidRangeStatus oMidRangeStatus;

   //Get the data using Helper function
   if(mClientHandlerFCAudioMgr)
      mClientHandlerFCAudioMgr->vCopyDatatoFIMessageWrapper(poMessage,oMidRangeStatus);
   log(&ControllerPlugin,DLT_LOG_INFO,"vResponse_SinkSndProp_MidRange,MidRange:",oMidRangeStatus.MidRange,",Resource:",
      oMidRangeStatus.Resource.enType,",Channel:",oMidRangeStatus.Channel.enType);

   if(!mListMainSinkSoundPropertyChanges.empty())
   {
      std::vector<rMainSinkSoundPropertySet>::iterator it(mListMainSinkSoundPropertyChanges.begin());
      for (; it != mListMainSinkSoundPropertyChanges.end(); ++it)
      {
         log(&ControllerPlugin,DLT_LOG_INFO,"vResponse_SinkSndProp_MidRange,received MidRange status,calling changeMainSinkSoundPropertyDB,type:",it->mainSoundProperty.type,",value:",it->mainSoundProperty.value);
         it->mainSoundProperty.value = oMidRangeStatus.MidRange;
         if(mControlReceiveInterface)
            mControlReceiveInterface->changeMainSinkSoundPropertyDB(it->mainSoundProperty, it->sinkID);
         mListMainSinkSoundPropertyChanges.erase(it);
         break;
      }
   }
   else{
      log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::processCCAResponse,received MidRange status,ignoring MidRange status....");
   }
}

/************************************************************************
 *FUNCTION     : vResponse_SinkSndProp_Loudness
 *DESCRIPTION  : to process LoudnessStatus response
 *PARAMETER    : amt_tclServiceData*
 *RETURN VALUE : void
 *HISTORY         :
 *20.09.2013   Rev 1.0     RBEI/ECG4 - Vyankatesh VD  Initial Revision
************************************************************************/
void CAmControlSenderBase::vResponse_SinkSndProp_Loudness(amt_tclServiceData* poMessage)
{
   log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::vResponse_SinkSndProp_Loudness,received Loudness status");
   midw_mascffi_tclMsgLoudnessStatus oLoudnessMessageStatus;

   //Get the data using Helper function
   if(mClientHandlerFCAudioMgr)
      mClientHandlerFCAudioMgr->vCopyDatatoFIMessageWrapper(poMessage,oLoudnessMessageStatus);
   log(&ControllerPlugin,DLT_LOG_INFO,"vResponse_SinkSndProp_Loudness received Status,Loudness:",oLoudnessMessageStatus.Loudness,
      ",Resource:",oLoudnessMessageStatus.Resource.enType,",Channel",oLoudnessMessageStatus.Channel.enType);

   std::vector<rMainSinkSoundPropertySet>::iterator it(mListMainSinkSoundPropertyChanges.begin());
   for (; it != mListMainSinkSoundPropertyChanges.end(); ++it)
   {
      log(&ControllerPlugin,DLT_LOG_INFO,"vResponse_SinkSndProp_Loudness,received Loudness,calling changeMainSinkSoundPropertyDB,type:",it->mainSoundProperty.type,",value:",it->mainSoundProperty.value);
      it->mainSoundProperty.value = oLoudnessMessageStatus.Loudness;
      if(mControlReceiveInterface)
         mControlReceiveInterface->changeMainSinkSoundPropertyDB(it->mainSoundProperty, it->sinkID);
      mListMainSinkSoundPropertyChanges.erase(it);
      break;
   }
}

/************************************************************************
 *FUNCTION     : vResponse_SinkSndProp_Balance
 *DESCRIPTION  : to process BalanceStatus response
 *PARAMETER    : amt_tclServiceData*
 *RETURN VALUE : void
 *HISTORY         :
 *20.09.2013   Rev 1.0     RBEI/ECG4 - Vyankatesh VD  Initial Revision
************************************************************************/
void CAmControlSenderBase::vResponse_SinkSndProp_Balance(amt_tclServiceData* poMessage)
{
   log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::vResponse_SinkSndProp_Balance,received Balance status");
   midw_mascffi_tclMsgBalanceStatus oBalanceMessageStatus;

   //Get the data using Helper function
   if(mClientHandlerFCAudioMgr)
      mClientHandlerFCAudioMgr->vCopyDatatoFIMessageWrapper(poMessage,oBalanceMessageStatus);
   log(&ControllerPlugin,DLT_LOG_INFO,"vResponse_SinkSndProp_Balance received Status,Balance:",oBalanceMessageStatus.Balance,
         ",Resource:",oBalanceMessageStatus.Resource.enType,",Channel",oBalanceMessageStatus.Channel.enType);

   std::vector<rMainSinkSoundPropertySet>::iterator it(mListMainSinkSoundPropertyChanges.begin());
   for (; it != mListMainSinkSoundPropertyChanges.end(); ++it)
   {
      log(&ControllerPlugin,DLT_LOG_INFO,"vResponse_SinkSndProp_Balance,received Balance status,calling changeMainSinkSoundPropertyDB,type:",it->mainSoundProperty.type,",value:",it->mainSoundProperty.value);
      it->mainSoundProperty.value = oBalanceMessageStatus.Balance;
      if(mControlReceiveInterface)
         mControlReceiveInterface->changeMainSinkSoundPropertyDB(it->mainSoundProperty, it->sinkID);
      mListMainSinkSoundPropertyChanges.erase(it);
      break;
   }
}

/************************************************************************
 *FUNCTION     : vResponse_SinkSndProp_Fader
 *DESCRIPTION  : to process FaderStatus response
 *PARAMETER    : amt_tclServiceData*
 *RETURN VALUE : void
 *HISTORY         :
 *20.09.2013   Rev 1.0     RBEI/ECG4 - Vyankatesh VD  Initial Revision
************************************************************************/
void CAmControlSenderBase::vResponse_SinkSndProp_Fader(amt_tclServiceData* poMessage)
{
   log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::vResponse_SinkSndProp_Fader,received Fader status");
   midw_mascffi_tclMsgFaderStatus oFaderMessageStatus;

   //Get the data using Helper function
   if(mClientHandlerFCAudioMgr)
      mClientHandlerFCAudioMgr->vCopyDatatoFIMessageWrapper(poMessage,oFaderMessageStatus);
   log(&ControllerPlugin,DLT_LOG_INFO,"vResponse_SinkSndProp_Fader received Status,Fader:",oFaderMessageStatus.Fader,
      ",Resource:",oFaderMessageStatus.Resource.enType,",Channel",oFaderMessageStatus.Channel.enType);

   std::vector<rMainSinkSoundPropertySet>::iterator it(mListMainSinkSoundPropertyChanges.begin());
   for (; it != mListMainSinkSoundPropertyChanges.end(); ++it)
   {
      log(&ControllerPlugin,DLT_LOG_INFO,"vResponse_SinkSndProp_Fader,received Fader status,calling changeMainSinkSoundPropertyDB,type:",it->mainSoundProperty.type,",value:",it->mainSoundProperty.value);
      it->mainSoundProperty.value = oFaderMessageStatus.Fader;
      if(mControlReceiveInterface)
         mControlReceiveInterface->changeMainSinkSoundPropertyDB(it->mainSoundProperty, it->sinkID);
      mListMainSinkSoundPropertyChanges.erase(it);
      break;
   }
}

/************************************************************************
 *FUNCTION     : vResponse_SinkSndProp_DynCompression
 *DESCRIPTION  : to process DynCompressionStatus response
 *PARAMETER    : amt_tclServiceData*
 *RETURN VALUE : void
 *HISTORY         :
 *20.09.2013   Rev 1.0     RBEI/ECG4 - Vyankatesh VD  Initial Revision
************************************************************************/
void CAmControlSenderBase::vResponse_SinkSndProp_DynCompression(amt_tclServiceData* poMessage)
{
   log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::vResponse_SinkSndProp_DynCompression,received Fader status");
   midw_mascffi_tclMsgDynamicCompressionStatus oDynCompressionMessageStatus;

   //Get the data using Helper function
   if(mClientHandlerFCAudioMgr)
      mClientHandlerFCAudioMgr->vCopyDatatoFIMessageWrapper(poMessage,oDynCompressionMessageStatus);
   log(&ControllerPlugin,DLT_LOG_INFO,"vResponse_SinkSndProp_DynCompression received Status,DynCompression:",oDynCompressionMessageStatus.DynCompression.enType,
      ",Resource:",oDynCompressionMessageStatus.Resource.enType);

   std::vector<rMainSinkSoundPropertySet>::iterator it(mListMainSinkSoundPropertyChanges.begin());
   for (; it != mListMainSinkSoundPropertyChanges.end(); ++it)
   {
      log(&ControllerPlugin,DLT_LOG_INFO,"vResponse_SinkSndProp_DynCompression,received DynCompression status,calling changeMainSinkSoundPropertyDB,type:",it->mainSoundProperty.type,",value:",it->mainSoundProperty.value);
      it->mainSoundProperty.value = oDynCompressionMessageStatus.DynCompression.enType;
      if(mControlReceiveInterface)
         mControlReceiveInterface->changeMainSinkSoundPropertyDB(it->mainSoundProperty, it->sinkID);
      mListMainSinkSoundPropertyChanges.erase(it);
      break;
   }

}

/************************************************************************
 *FUNCTION     : vResponse_SinkSndProp_EquPreset
 *DESCRIPTION  : to process EquPresetStatus response
 *PARAMETER    : amt_tclServiceData*
 *RETURN VALUE : void
 *HISTORY         :
 *20.09.2013   Rev 1.0     RBEI/ECG4 - Vyankatesh VD  Initial Revision
************************************************************************/
void CAmControlSenderBase::vResponse_SinkSndProp_EquPreset(amt_tclServiceData* poMessage)
{
   log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::vResponse_SinkSndProp_EquPreset,received Fader status");
   midw_mascffi_tclMsgEqualiserPresetStatus oEquPresetMessageStatus;

   //Get the data using Helper function
   if(mClientHandlerFCAudioMgr)
      mClientHandlerFCAudioMgr->vCopyDatatoFIMessageWrapper(poMessage,oEquPresetMessageStatus);
   log(&ControllerPlugin,DLT_LOG_INFO,"vResponse_SinkSndProp_EquPreset received Status,EquPreset:",oEquPresetMessageStatus.EquPreset.enType,
      ",Resource:",oEquPresetMessageStatus.Resource.enType);

   std::vector<rMainSinkSoundPropertySet>::iterator it(mListMainSinkSoundPropertyChanges.begin());
   for (; it != mListMainSinkSoundPropertyChanges.end(); ++it)
   {
      log(&ControllerPlugin,DLT_LOG_INFO,"vResponse_SinkSndProp_EquPreset,received EquPreset status,calling changeMainSinkSoundPropertyDB,type:",it->mainSoundProperty.type,",value:",it->mainSoundProperty.value);
      it->mainSoundProperty.value = oEquPresetMessageStatus.EquPreset.enType;
      if(mControlReceiveInterface)
         mControlReceiveInterface->changeMainSinkSoundPropertyDB(it->mainSoundProperty, it->sinkID);
      mListMainSinkSoundPropertyChanges.erase(it);
      break;
   }
}

/************************************************************************
 *FUNCTION     : vResponse_SinkSndProp_Gala
 *DESCRIPTION  : to process GalaStatus response
 *PARAMETER    : amt_tclServiceData*
 *RETURN VALUE : void
 *HISTORY         :
 *20.09.2013   Rev 1.0     RBEI/ECG4 - Vyankatesh VD  Initial Revision
************************************************************************/
void CAmControlSenderBase::vResponse_SinkSndProp_Gala(amt_tclServiceData* poMessage)
{
   log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::vResponse_SinkSndProp_Gala,received SDVC/Gala status");
   midw_mascffi_tclMsgGalaStatus oGalaMessageStatus;

   //Get the data using Helper function
   if(mClientHandlerFCAudioMgr)
      mClientHandlerFCAudioMgr->vCopyDatatoFIMessageWrapper(poMessage,oGalaMessageStatus);
   log(&ControllerPlugin,DLT_LOG_INFO,"vResponse_SinkSndProp_Gala received Status,EquPreset:",oGalaMessageStatus.GalaLevel.enType,
      ",Resource:",oGalaMessageStatus.Resource.enType);

   std::vector<rMainSinkSoundPropertySet>::iterator it(mListMainSinkSoundPropertyChanges.begin());
   for (; it != mListMainSinkSoundPropertyChanges.end(); ++it)
   {
      it->mainSoundProperty.value = oGalaMessageStatus.GalaLevel.enType;
      log(&ControllerPlugin,DLT_LOG_INFO,"vResponse_SinkSndProp_Gala,received SDVC/Gala status,calling changeMainSinkSoundPropertyDB,type:",it->mainSoundProperty.type,",value:",it->mainSoundProperty.value);
      if(mControlReceiveInterface)
         mControlReceiveInterface->changeMainSinkSoundPropertyDB(it->mainSoundProperty, it->sinkID);
      mListMainSinkSoundPropertyChanges.erase(it);
      break;
   }
}

/************************************************************************
 *FUNCTION     : vResponse_SinkSndProp_Optimization
 *DESCRIPTION  : to process OptimizationStatus response
 *PARAMETER    : amt_tclServiceData*
 *RETURN VALUE : void
 *HISTORY         :
 *20.09.2013   Rev 1.0     RBEI/ECG4 - Vyankatesh VD  Initial Revision
************************************************************************/
void CAmControlSenderBase::vResponse_SinkSndProp_Optimization(amt_tclServiceData* poMessage)
{
   log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::vResponse_SinkSndProp_Optimization,received SndOptimization status");
   midw_mascffi_tclMsgOptimizationStatus oSndProOptStatusMessage;

   //Get the data using Helper function
   if(mClientHandlerFCAudioMgr)
      mClientHandlerFCAudioMgr->vCopyDatatoFIMessageWrapper(poMessage,oSndProOptStatusMessage);
   log(&ControllerPlugin,DLT_LOG_INFO,"vResponse_SinkSndProp_Optimization received Status,Optimization:",oSndProOptStatusMessage.Optimization.enType,
      ",Resource:",oSndProOptStatusMessage.Resource.enType);

   std::vector<rMainSinkSoundPropertySet>::iterator it(mListMainSinkSoundPropertyChanges.begin());
   for (; it != mListMainSinkSoundPropertyChanges.end(); ++it)
   {
      it->mainSoundProperty.value = oSndProOptStatusMessage.Optimization.enType;
      log(&ControllerPlugin,DLT_LOG_INFO,"vResponse_SinkSndProp_Optimization,received SndOptimization status,calling changeMainSinkSoundPropertyDB,type:",it->mainSoundProperty.type,",value:",it->mainSoundProperty.value);
      if(mControlReceiveInterface)
         mControlReceiveInterface->changeMainSinkSoundPropertyDB(it->mainSoundProperty, it->sinkID);
      mListMainSinkSoundPropertyChanges.erase(it);
      break;
   }
}

/************************************************************************
 *FUNCTION     : vResponse_SrcSndProp_MicPhoneConn
 *DESCRIPTION  : to process MicPhoneConnStatus response
 *PARAMETER    : amt_tclServiceData*
 *RETURN VALUE : void
 *HISTORY         :
 *20.09.2013   Rev 1.0     RBEI/ECG4 - Vyankatesh VD  Initial Revision
************************************************************************/
void CAmControlSenderBase::vResponse_SrcSndProp_MicPhoneConn(amt_tclServiceData* poMessage)
{
   log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::vResponse_SrcSndProp_MicPhoneConn,received micPhoneConn status");
   midw_mascffi_tclMsgMicrophoneConnectionStatusStatus oMicConnStatusMessage;

   //Get the data using Helper function
   if(mClientHandlerFCAudioMgr)
      mClientHandlerFCAudioMgr->vCopyDatatoFIMessageWrapper(poMessage,oMicConnStatusMessage);
   log(&ControllerPlugin,DLT_LOG_INFO,"vResponse_SrcSndProp_MicPhoneConn received Status,u16MicrophoneCurrent:",oMicConnStatusMessage.u16MicrophoneCurrent,
      ",e8MicrophoneConnectionStatus:",oMicConnStatusMessage.e8MicrophoneConnectionStatus.enType);

   std::vector<rMainSourceSoundPropertySet>::iterator it(mListMainSourceSoundPropertyChanges.begin());
   for (; it != mListMainSourceSoundPropertyChanges.end(); ++it)
   {
      it->mainSoundProperty.value = oMicConnStatusMessage.u16MicrophoneCurrent;
      log(&ControllerPlugin,DLT_LOG_INFO,"vResponse_SrcSndProp_MicPhoneConn,received micPhoneConn status,calling changeMainSourceSoundPropertyDB,type:",it->mainSoundProperty.type,",value:",it->mainSoundProperty.value);
      if(mControlReceiveInterface)
         mControlReceiveInterface->changeMainSourceSoundPropertyDB(it->mainSoundProperty, it->sourceID);
      mListMainSourceSoundPropertyChanges.erase(it);
      break;
   }
}

/************************************************************************
 *FUNCTION     : vResponse_SrcSndProp_AuxInLevel
 *DESCRIPTION  : to process AuxInLevelStatus response
 *PARAMETER    : amt_tclServiceData*
 *RETURN VALUE : void
 *HISTORY         :
 *20.09.2013   Rev 1.0     RBEI/ECG4 - Vyankatesh VD  Initial Revision
************************************************************************/
void CAmControlSenderBase::vResponse_SrcSndProp_AuxInLevel(amt_tclServiceData* poMessage)
{
   log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::vResponse_SrcSndProp_AuxInLevel,received AuxInLevel status");
   midw_mascffi_tclMsgAuxInLevelStatus oAuxInLevelStatusMessage;

   //Get the data using Helper function
   if(mClientHandlerFCAudioMgr)
      mClientHandlerFCAudioMgr->vCopyDatatoFIMessageWrapper(poMessage,oAuxInLevelStatusMessage);
   log(&ControllerPlugin,DLT_LOG_INFO,"vResponse_SrcSndProp_AuxInLevel received Status,AuxInLevel:",
         oAuxInLevelStatusMessage.AuxInLevel.enType);

   std::vector<rMainSourceSoundPropertySet>::iterator it(mListMainSourceSoundPropertyChanges.begin());
   for (; it != mListMainSourceSoundPropertyChanges.end(); ++it)
   {
      it->mainSoundProperty.value = oAuxInLevelStatusMessage.AuxInLevel.enType;
      log(&ControllerPlugin,DLT_LOG_INFO,"vResponse_SrcSndProp_AuxInLevel,received AuxInLevel status,calling changeMainSourceSoundPropertyDB,type:",it->mainSoundProperty.type,",value:",it->mainSoundProperty.value);
      if(mControlReceiveInterface)
         mControlReceiveInterface->changeMainSourceSoundPropertyDB(it->mainSoundProperty, it->sourceID);
      mListMainSourceSoundPropertyChanges.erase(it);
      break;
   }

}

/************************************************************************
 *FUNCTION     : vResponse_SysSndProp_BeepStatus
 *DESCRIPTION  : to process BeepStatus response
 *PARAMETER    : amt_tclServiceData*
 *RETURN VALUE : void
 *HISTORY         :
 *20.09.2013   Rev 1.0     RBEI/ECG4 - Vyankatesh VD  Initial Revision
************************************************************************/
void CAmControlSenderBase::vResponse_SysSndProp_BeepStatus(amt_tclServiceData* poMessage)
{
   log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::vResponse_SysSndProp_BeepStatus,received Beep Status");
   midw_mascffi_tclMsgBeepStatusStatus oBeepStatusMessage;

   //Get the data using Helper function
   if(mClientHandlerFCAudioMgr)
      mClientHandlerFCAudioMgr->vCopyDatatoFIMessageWrapper(poMessage,oBeepStatusMessage);
   log(&ControllerPlugin,DLT_LOG_INFO,"vResponse_SysSndProp_BeepStatus received Status,BeepStatus:",oBeepStatusMessage.BeepStatus.enType,
      ",Resource:",oBeepStatusMessage.Resource.enType);

   std::vector<am_SystemProperty_s>::iterator it(mListMainSystemSoundPropertyChanges.begin());
   for (; it != mListMainSystemSoundPropertyChanges.end(); ++it)
   {
      it->value = oBeepStatusMessage.BeepStatus.enType;
      log(&ControllerPlugin,DLT_LOG_INFO,"vResponse_SysSndProp_BeepStatus,received BeepStatus status,calling changeMainSinkSoundPropertyDB,type:",it->type,",value:",it->value);
      if(mControlReceiveInterface)
         mControlReceiveInterface->changeSystemPropertyDB(*it);
      mListMainSystemSoundPropertyChanges.erase(it);
      break;
   }
}

/************************************************************************
 *FUNCTION     : vResponse_SysSndProp_FaderActivation
 *DESCRIPTION  : to process FaderActivationStatus response
 *PARAMETER    : amt_tclServiceData*
 *RETURN VALUE : void
 *HISTORY         :
 *20.09.2013   Rev 1.0     RBEI/ECG4 - Vyankatesh VD  Initial Revision
************************************************************************/
void CAmControlSenderBase::vResponse_SysSndProp_FaderActivation(amt_tclServiceData* poMessage)
{
   log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::vResponse_SysSndProp_FaderActivation,received FaderActivation Status");
   midw_mascffi_tclMsgFaderActivationStatusStatus oFaderActStatusMessage;

   //Get the data using Helper function
   if(mClientHandlerFCAudioMgr)
      mClientHandlerFCAudioMgr->vCopyDatatoFIMessageWrapper(poMessage,oFaderActStatusMessage);
   log(&ControllerPlugin,DLT_LOG_INFO,"vResponse_SysSndProp_FaderActivation received Status,FaderStatus:",
         oFaderActStatusMessage.FaderStatus.enType);

   std::vector<am_SystemProperty_s>::iterator it(mListMainSystemSoundPropertyChanges.begin());
   for (; it != mListMainSystemSoundPropertyChanges.end(); ++it)
   {
      it->value = oFaderActStatusMessage.FaderStatus.enType;
      log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::vResponse_SysSndProp_FaderActivation,received FaderActivation status,calling changeSystemPropertyDB,type:",it->type,",value:",it->value);
      if(mControlReceiveInterface)
         mControlReceiveInterface->changeSystemPropertyDB(*it);
      mListMainSystemSoundPropertyChanges.erase(it);
      break;
   }

}
/************************************************************************
 *FUNCTION     : vResponse_SysSndProp_VolumeMode
 *DESCRIPTION  : to process VolumeMode response
 *PARAMETER    : amt_tclServiceData*
 *RETURN VALUE : void
 *HISTORY         :
 *20.09.2013   Rev 1.0     RBEI/ECG4 - Vyankatesh VD  Initial Revision
************************************************************************/
void CAmControlSenderBase::vResponse_SysSndProp_VolumeMode(amt_tclServiceData* poMessage)
{
   log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::vResponse_SysSndProp_VolumeMode,received VolumeMode Status");
   midw_mascffi_tclMsgVolumeModeStatusStatus oVolumeModeStatus;

   //Get the data using Helper function
   if(mClientHandlerFCAudioMgr)
      mClientHandlerFCAudioMgr->vCopyDatatoFIMessageWrapper(poMessage,oVolumeModeStatus);
   log(&ControllerPlugin,DLT_LOG_INFO,"vResponse_SysSndProp_VolumeMode received,VolumeMode:",oVolumeModeStatus.VolumeMode.enType,
      ",Resource:",oVolumeModeStatus.Resource.enType,",Channel:",oVolumeModeStatus.Channel.enType);

   std::vector<am_SystemProperty_s>::iterator it(mListMainSystemSoundPropertyChanges.begin());
   for (; it != mListMainSystemSoundPropertyChanges.end(); ++it)
   {
      it->value = oVolumeModeStatus.VolumeMode.enType;
      log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::vResponse_SysSndProp_VolumeMode,received VolumeMode status,calling changeSystemPropertyDB,type:",it->type,",value:",it->value);
      if(mControlReceiveInterface)
         mControlReceiveInterface->changeSystemPropertyDB(*it);
      mListMainSystemSoundPropertyChanges.erase(it);
      break;
   }
}
/************************************************************************
* FUNCTION        : processIncCcaResponse
* DESCRIPTION     : to fetch cca response from queue and process it.
* PARAMETER       : NA
* RETURNVALUE     : void
* HISTORY         :
* 20.08.2013   Rev 1.0     RBEI/ECG4 - Vyankatesh VD  Initial Revision
***********************************************************************/
void CAmControlSenderBase::processIncCcaResponse()
{
   log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::processCCAResponse,entered in processCCAResponse");

   if(NULL == mControlReceiveShadowInterface)
      return;

   log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::processCCAResponse,found entry in m_ListCCAResponseMsg");

   amt_tclServiceData* poMessage = mControlReceiveShadowInterface->remove_queue();

   if(NULL == poMessage){
      log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::processCCAResponse, poMessage is NULL");
      return;
   }

   switch(poMessage->u16GetFunctionID())
   {
      case MIDW_AUDIO_ROUTE_MGRFI_C_U16_SOURCEAVAILABILITY:
         vResponse_SourceAvailability(poMessage);
         break;
      //Source Change
      case MIDW_AUDIO_ROUTE_MGRFI_C_U16_P_SOURCEROUTEACTIVITY: /*RESPONSE_SOURCE_ROUTE_ACTIVITY*/
      {
     log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::processCCAResponse, calling vResponse_SourceChange");
     vResponse_SourceChange(poMessage);

       if((mControlReceiveShadowInterface->m_bUserRequest)
         && (REQUEST_MUTE_DEMUTE == mControlReceiveShadowInterface->enRequestType))
         {
          log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::processCCAResponse, calling vResponse_Mute");
          mControlReceiveShadowInterface->m_bUserRequest = false;
          mControlReceiveShadowInterface->enRequestType = REQUEST_UNKNOWN;
          vResponse_Mute(poMessage);
         }
         break;
      }
      //Source Change
      case MIDW_AUDIO_ROUTE_MGRFI_C_U16_REQUESTAUDIOROUTE:
      {
        vResponse_AudioRoute(poMessage);
         break;
      }
      /* volume status
       * Not required to use setSinkSoundProperty for thread switching for volume status
       */
      case MIDW_MASCFFI_C_U16_VOLUME :
      {
         vResponse_Volume(poMessage);
         break;
      }
      //Mute Method Result
      case MIDW_MASCFFI_C_U16_MUTE :
      {
         vResponse_Mute(poMessage);
         break;
      }
      //Bass Status
      case MIDW_MASCFFI_C_U16_BASS :
      {
         vResponse_SinkSndProp_Bass(poMessage);
         break;
      }
      //Treble Status
      case MIDW_MASCFFI_C_U16_TREBLE :
      {
         vResponse_SinkSndProp_Treble(poMessage);
         break;
      }
      //MidRange Status
      case MIDW_MASCFFI_C_U16_MIDRANGE :
      {
         vResponse_SinkSndProp_MidRange(poMessage);
         break;
      }
      //Loudness Status
      case MIDW_MASCFFI_C_U16_LOUDNESS :
      {
         vResponse_SinkSndProp_Loudness(poMessage);
         break;
      }
      //Balance Status
      case MIDW_MASCFFI_C_U16_BALANCE :
      {
         vResponse_SinkSndProp_Balance(poMessage);
         break;
      }
      //Balance Status
      case MIDW_MASCFFI_C_U16_FADER :
      {
         vResponse_SinkSndProp_Fader(poMessage);
         break;
      }
      //DynamicCompression Status
      case MIDW_MASCFFI_C_U16_DYNAMICCOMPRESSION :
      {
         vResponse_SinkSndProp_DynCompression(poMessage);
         break;
      }
      //Equalizer Preset Status
      case MIDW_MASCFFI_C_U16_EQUALISERPRESET :
      {
         vResponse_SinkSndProp_EquPreset(poMessage);
         break;
      }
      //Equalizer Preset Status
      case MIDW_MASCFFI_C_U16_GALA :
      {
         vResponse_SinkSndProp_Gala(poMessage);
         break;
      }
      //Micro Phone Connection Status
      case MIDW_MASCFFI_C_U16_MICROPHONECONNECTIONSTATUS :
      {
         vResponse_SrcSndProp_MicPhoneConn(poMessage);
         break;
      }
      //Sound Optimization Status
      case MIDW_MASCFFI_C_U16_OPTIMIZATION :
      {
         vResponse_SinkSndProp_Optimization(poMessage);
         break;
      }
      //AuxInLevel Status
      case MIDW_MASCFFI_C_U16_AUXINLEVEL :
      {
         vResponse_SrcSndProp_AuxInLevel(poMessage);
         break;
      }
      //Beep Status  System Property
      case MIDW_MASCFFI_C_U16_BEEPSTATUS :
      {
         vResponse_SysSndProp_BeepStatus(poMessage);
         break;
      }
      //FaderActivation Status System Property
      case MIDW_MASCFFI_C_U16_FADERACTIVATIONSTATUS :
      {
         vResponse_SysSndProp_FaderActivation(poMessage);
         break;
      }
      //VolumeMode Status System Property
      case MIDW_MASCFFI_C_U16_VOLUMEMODESTATUS :
      {
         vResponse_SysSndProp_VolumeMode(poMessage);
         break;
      }
      default :
      {
         logInfo("CAmControlSenderBase::processCCAResponse, Unknown FID");
         break;
      }
   }//End Switch

   poMessage->bDelete();
   delete poMessage;

}
/************************************************************************
* FUNCTION     : cbAckSetSourceSoundProperties
* DESCRIPTION     : ack for SetSourceSoundProperties
* PARAMETER    : const am_Handle_s , const am_Error_e
* RETURNVALUE     : void
* HISTORY         : GENIVI
************************************************************************/
void CAmControlSenderBase::cbAckSetSourceSoundProperties(const am_Handle_s handle, const am_Error_e error)
{
   (void) handle;
   (void) error;
}

/************************************************************************
* FUNCTION     : cbAckSetSinkSoundProperties
* DESCRIPTION     : ack for SetSinkSoundProperties
* PARAMETER    : const am_Handle_s , const am_Error_e
* RETURNVALUE     : void
* HISTORY         : GENIVI
************************************************************************/
void CAmControlSenderBase::cbAckSetSinkSoundProperties(const am_Handle_s handle, const am_Error_e error)
{
    (void) error;
    (void) handle;
}

/************************************************************************
* FUNCTION     : setControllerRundown
* DESCRIPTION     : to unload plugin
* PARAMETER    : void
* RETURNVALUE     : void
* HISTORY         : GENIVI
************************************************************************/
void CAmControlSenderBase::setControllerRundown(const int16_t isignal)
{
    //Send ack to Daemon
   logInfo("CAmControlSenderBase::setControllerRundown() was called with signal=",isignal);
   if(mControlReceiveInterface)
      mControlReceiveInterface->confirmControllerRundown(E_OK);

}

/************************************************************************
* FUNCTION     : getConnectionFormatChoice
* DESCRIPTION     : This function is used by the routing algorithm to retrieve a priorized list of connectionFormats from the Controller.
* PARAMETER    : const am_sourceID_t, const am_sinkID_t, const am_Route_s , const std::vector<am_ConnectionFormat_e> , std::vector<am_ConnectionFormat_e>
* RETURNVALUE     : am_Error_e
* HISTORY         : GENIVI
************************************************************************/
am_Error_e CAmControlSenderBase::getConnectionFormatChoice(const am_sourceID_t sourceID, const am_sinkID_t sinkID, const am_Route_s listRoute, const std::vector<am_ConnectionFormat_e> listPossibleConnectionFormats, std::vector<am_ConnectionFormat_e> & listPrioConnectionFormats)
{
    (void) sourceID;
    (void) sinkID;
    (void) listRoute;
    //ok, this is cheap. In a real product you have your preferences, right?
    listPrioConnectionFormats = listPossibleConnectionFormats;
    return (E_OK);
}

/************************************************************************
* FUNCTION     : getInterfaceVersion
* DESCRIPTION     : to get interface version
* PARAMETER    : std::string &
* RETURNVALUE     : void
* HISTORY         : GENIVI
************************************************************************/
void CAmControlSenderBase::getInterfaceVersion(std::string & version) const
{
    version = ControlSendVersion;
}

/************************************************************************
* FUNCTION     : confirmCommandReady
* DESCRIPTION     : confirmation for command ready
* PARAMETER    : void
* RETURNVALUE     : void
* HISTORY         : GENIVI
************************************************************************/
void CAmControlSenderBase::confirmCommandReady(const am_Error_e error)
{
    (void) error;
   logInfo("CAmControlSenderBase::confirmCommandReady() was called with error=",error);
   logInfo("ControlSenderPlugin got Command Ready confirmed");
}

/************************************************************************
* FUNCTION     : confirmRoutingReady
* DESCRIPTION     : confirmation for Routing ready
* PARAMETER    : void
* RETURNVALUE     : void
* HISTORY         : GENIVI
************************************************************************/
void CAmControlSenderBase::confirmRoutingReady(const am_Error_e error)
{
   (void) error;

   if(NULL == mControlReceiveShadowInterface)
      return;

   log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::confirmRoutingReady,got routing Ready confirmation,calling confirmControllerReady");

   mControlReceiveShadowInterface->m_bRoutingReady = true;

    //check if both service avaialble and command ready confirmation then call confirmControllerReady
    if((true == mControlReceiveShadowInterface->m_bServiceReady_fcAudioMgr) &&
         (true == mControlReceiveShadowInterface->m_bServiceReady_fcAudioRouteMgr))
    {

      if(false == mControlReceiveShadowInterface->m_bControllerReady)
      {
         mControlReceiveShadowInterface->m_bControllerReady = true;
         log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::confirmRoutingReady,both service fc_audiomgr and fc_audioroutemgr are available. \
                calling confirmControllerReady");

         mControlReceiveShadowInterface->confirmControllerReady(E_OK);
         InitStackManager();

      }
      else{
         log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::confirmRoutingReady,both service fc_audiomgr and fc_audioroutemgr are available. \
                Already called confirmControllerReady");
      }
    }
    else
    {
      log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::confirmRoutingReady,services fc_audiomgr and fc_audioroutemgr are not available. \
             Waiting for services to be available... ");
    }
}

/************************************************************************
* FUNCTION     : confirmCommandRundown
* DESCRIPTION     : confirmation for Command  rundown
* PARAMETER    : void
* RETURNVALUE     : void
* HISTORY         : GENIVI
************************************************************************/
void CAmControlSenderBase::confirmCommandRundown(const am_Error_e error)
{
   (void) error;
   logInfo("CAmControlSenderBase::confirmCommandRundown() was called with error=",error);
   log(&ControllerPlugin,DLT_LOG_INFO,"ControlSenderPlugin got Routing Rundown confirmed");
}

/************************************************************************
* FUNCTION     : confirmRoutingRundown
* DESCRIPTION     : confirmation for Routing  rundown
* PARAMETER    : void
* RETURNVALUE     : void
* HISTORY         : GENIVI
************************************************************************/
void CAmControlSenderBase::confirmRoutingRundown(const am_Error_e error)
{
    (void) error;
   logInfo("CAmControlSenderBase::confirmRoutingRundown() was called with error=",error);
   log(&ControllerPlugin,DLT_LOG_INFO,"ControlSenderPlugin got Command Rundown confirmed");
}

/************************************************************************
* FUNCTION     : connect
* DESCRIPTION  : connect a source to a sink
* PARAMETER    : sourceID, sinkID, mainConnectionID
* RETURNVALUE  : void
* HISTORY      : GENIVI
************************************************************************/
void CAmControlSenderBase::connect(am_sourceID_t sourceID, am_sinkID_t sinkID, am_mainConnectionID_t mainConnectionID) const
{
   log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::connect called source id",sourceID,"sink id",sinkID,"main connection id",mainConnectionID);
}

/************************************************************************
* FUNCTION     : disconnect
* DESCRIPTION  : Disconnect an established connection
* PARAMETER    : connectionID
* RETURNVALUE  : void
* HISTORY      : GENIVI
************************************************************************/
void CAmControlSenderBase::disconnect(am_mainConnectionID_t ) const
{

}

/************************************************************************
* FUNCTION     : callStateFlowHandler
* DESCRIPTION  : State Machine for the Audio Management
* PARAMETER    : none
* RETURNVALUE  : void
* HISTORY      : GENIVI
************************************************************************/
void CAmControlSenderBase::callStateFlowHandler()
{
    logInfo("CAmControlSenderBase::callStateFlowHandler() called, state ",mStateflow);

    switch(mStateflow)
    {
    case SF_CONNECT:
        callConnectHandler();
        break;
    default:
        break;

    }
}

/************************************************************************
* FUNCTION     : callConnectHandler
* DESCRIPTION  : State Machine for the Connect/Disconnect operations
*             Audio Manager has a very simple state machine now.
*                 On getting a new connection request now, remove the
*                 old connection and then connect the new one
* PARAMETER    : none
* RETURNVALUE  : void
* HISTORY      : GENIVI
************************************************************************/
void CAmControlSenderBase::callConnectHandler()
{
   log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::callConnectHandler() called, state ",mConnectSf);

    switch (mConnectSf)
    {
    case SFC_DISCONNECT:
      mConnectSf = SFC_CONNECT;
        disconnect(mConnectData.currentMainConnection);
        break;
    case SFC_CONNECT:
      mStateflow=SF_NONE;
      mConnectSf = SFC_FINISHED;
        connect(mConnectData.sourceID, mConnectData.sinkID, mConnectData.newMainConnection);
        break;
    default:
        break;
    }
}
void CAmControlSenderBase::hookSystemNodeStateChanged(const NsmNodeState_e NodeStateId)
{
    (void) NodeStateId;
    //here you can process informations about the notestate
}

void CAmControlSenderBase::hookSystemNodeApplicationModeChanged(const NsmApplicationMode_e ApplicationModeId)
{
    (void) ApplicationModeId;
}


void CAmControlSenderBase::cbAckSetSinkNotificationConfiguration(const am_Handle_s handle, const am_Error_e error)
{
    (void) handle;
    (void) error;
}

void CAmControlSenderBase::cbAckSetSourceNotificationConfiguration(const am_Handle_s handle, const am_Error_e error)
{
    (void) handle;
    (void) error;
}


NsmErrorStatus_e CAmControlSenderBase::hookSystemLifecycleRequest(const uint32_t Request, const uint32_t RequestId)
{
    (void) Request;
    (void) RequestId;
    logInfo("CAmControlSenderBase::hookSystemLifecycleRequest request=",Request," requestID=",RequestId);
    return (NsmErrorStatus_Error);
}

am_Error_e CAmControlSenderBase::hookSystemUpdateSink(const am_sinkID_t sinkID, const am_sinkClass_t sinkClassID, const std::vector<am_SoundProperty_s>& listSoundProperties, const std::vector<am_ConnectionFormat_e>& listConnectionFormats, const std::vector<am_MainSoundProperty_s>& listMainSoundProperties)
{
    (void) sinkID;
    (void) sinkClassID;
    (void) listMainSoundProperties;
    (void) listConnectionFormats;
    (void) listSoundProperties;
    return (E_NOT_USED);
}

am_Error_e CAmControlSenderBase::hookSystemUpdateSource(const am_sourceID_t sourceID, const am_sourceClass_t sourceClassID, const std::vector<am_SoundProperty_s>& listSoundProperties, const std::vector<am_ConnectionFormat_e>& listConnectionFormats, const std::vector<am_MainSoundProperty_s>& listMainSoundProperties)
{
    (void) sourceID;
    (void) sourceClassID;
    (void) listSoundProperties;
    (void) listMainSoundProperties;
    (void) listConnectionFormats;
    return (E_NOT_USED);
}

am_Error_e CAmControlSenderBase::hookSystemUpdateGateway(const am_gatewayID_t gatewayID, const std::vector<am_ConnectionFormat_e>& listSourceConnectionFormats, const std::vector<am_ConnectionFormat_e>& listSinkConnectionFormats, const std::vector<bool>& convertionMatrix)
{
    (void) gatewayID;
    (void) listSourceConnectionFormats;
    (void) listSinkConnectionFormats;
    (void) convertionMatrix;
    return (E_NOT_USED);
}

void CAmControlSenderBase::cbAckSetVolumes(const am_Handle_s handle, const std::vector<am_Volumes_s>& listVolumes, const am_Error_e error)
{
    (void) handle;
    (void) listVolumes;
    (void) error;
}

void CAmControlSenderBase::hookSinkNotificationDataChanged(const am_sinkID_t sinkID, const am_NotificationPayload_s& payload)
{
    (void) sinkID;
    (void) payload;
}

void CAmControlSenderBase::hookSourceNotificationDataChanged(const am_sourceID_t sourceID, const am_NotificationPayload_s& payload)
{
    (void) sourceID;
    (void) payload;
}

am_Error_e CAmControlSenderBase::hookUserSetMainSinkNotificationConfiguration(const am_sinkID_t sinkID, const am_NotificationConfiguration_s& notificationConfiguration)
{
    (void) sinkID;
    (void) notificationConfiguration;
    return (E_NOT_USED);
}

am_Error_e CAmControlSenderBase::hookUserSetMainSourceNotificationConfiguration(const am_sourceID_t sourceID, const am_NotificationConfiguration_s& notificationConfiguration)
{
    (void) sourceID;
    (void) notificationConfiguration;
    return (E_NOT_USED);
}

void CAmControlSenderBase::hookSystemSessionStateChanged(const std::string& sessionName, const NsmSeat_e seatID, const NsmSessionState_e sessionStateID)
{
    (void) sessionName;
    (void) seatID;
    (void) sessionStateID;
}

/************************************************************************
* FUNCTION     : cbAckCCAMessageReceived
* DESCRIPTION  : Callback function to receive inc cca masg from gateway
*             will be called to swith the the thread context
*                 from inc to genivi
* PARAMETER    : const am_Handle_s handle, const am_Error_e error
* RETURNVALUE  : void
* HISTORY      :
*              10.09.12 Vyankatesh VD  Initial Revision.
************************************************************************/
void CAmControlSenderBase::cbAckCCAMessageReceived()
{

   log(&ControllerPlugin,DLT_LOG_INFO,"Entered in CAmControlSenderBase::cbAckCCAMessageReceived.....");

   if(NULL == mControlReceiveShadowInterface)
      return;


   if(!mControlReceiveShadowInterface->m_ListCCAResponseMsg.empty())
   {
      log(&ControllerPlugin,DLT_LOG_INFO,"CAmControlSenderBase::cbAckCCAMessageReceived,found CCA msg in m_ListCCAResponseMsg");

      processIncCcaResponse();

   }
   else
   {
      log(&ControllerPlugin,DLT_LOG_INFO,"Entered in CAmControlSenderBase::cbAckCCAMessageReceived, m_ListCCAResponseMsg is empty");
   }
}
